#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "gga.h"


/* the control module is the central dispatching unit...
 * it is responsible for the main select() loop as well as
 * reading and interpretting commands read from cmd,
 * including maintaining the structures that dispwav, disptrack, and snd
 * use for output.
 */

static int logfd = -1;
static void log_write(char *buf);

static unsigned read_time(char *s);
static void read_vol(char *s, unsigned *lvol, unsigned *rvol);
static unsigned read_freq(char *s);
static int scan_timespec(int argc, char **argv, int ind,
		unsigned *start, unsigned *stop);
static int scan_sourcespec(int argc, char **argv, int ind,
                struct waveform_rec **wav,
		unsigned *start, unsigned *stop);
static int scan_trackspec(int argc, char **argv, int ind,
                struct track_rec **track,
		unsigned *start, unsigned *stop);
static int scan_eventspec(int argc, char **argv, int ind, struct event_rec *e);


/* if possible, this reads and executes 'startup' file */
int
control_init(void)
{
	FILE *f;
	char buf[CMD_BUFLEN];

	f = fopen(STARTUP_FILE,"r");
	if (!f && errno == ENOENT) {
		goto skip_startup;
	}
	if (!f) {
		perror(STARTUP_FILE);
		goto skip_startup;
	}

	cmd_write("reading "STARTUP_FILE"\n");

	while (fgets(buf,CMD_BUFLEN,f)) {
		buf[strcspn(buf,"\r\n")] = 0;
		control_command(buf);
	}
	fclose(f);

skip_startup:
	logfd = open(LOG_FILE,O_WRONLY|O_APPEND|O_CREAT,0666);
	{
		time_t t;
		log_write("# log started ");
		time(&t);
		log_write(asctime(localtime(&t)));
		log_write("\n");
	}

	return 0;
}

/* the main select() loop, throws events to where they beint */
int
control_loop(void)
{
	fd_set rfds,wfds;
	int n;
	int max_fd = 0;

	if (snd_fd > max_fd) max_fd = snd_fd;
	if (cmd_fd > max_fd) max_fd = cmd_fd;
	if (x_fd > max_fd) max_fd = x_fd;

	for (;;) {
		/* make sure X has everything needed to reply.. */
		x_flush();

		FD_ZERO(&rfds);
		FD_ZERO(&wfds);

		if (snd_ready)
			FD_SET(snd_fd,&wfds);
		FD_SET(cmd_fd,&rfds);
		FD_SET(x_fd,&rfds);

		n = select(max_fd+1,&rfds,&wfds,NULL,NULL);
		if (n < 0) {
			if (errno == EINTR) {
				continue;
			}
			perror("select");
			return -1;
		}
		if (snd_ready && FD_ISSET(snd_fd,&wfds)) {
			snd_doit();
		}
		if (FD_ISSET(cmd_fd,&rfds)) {
			cmd_read();
		}
		if (FD_ISSET(x_fd,&rfds)) {
			x_event();
		}
	}
	return 0;
}


static void
log_write(char *buf)
{
	if (logfd < 0)
		return;
	write(logfd, buf, strlen(buf));
}


static void cc_load(int argc, char **argv);
static void cc_save(int argc, char **argv);
static void cc_copy(int argc, char **argv);
static void cc_filter(int argc, char **argv);
static void cc_write(int argc, char **argv);
static void cc_generate(int argc, char **argv);
static void cc_source(int argc, char **argv);
static void cc_play(int argc, char **argv);
static void cc_stop(int argc, char **argv);
static void cc_add(int argc, char **argv);
static void cc_edit(int argc, char **argv);
static void cc_drop(int argc, char **argv);
static void cc_setlength(int argc, char **argv);
static void cc_set(int argc, char **argv);
static void cc_info(int argc, char **argv);
static void cc_nop(int argc, char **argv);
static void cc_quit(int argc, char **argv);

struct control_command_rec {
	char *name;
	char *short_name;
	void (*func)(int argc,char **argv);
	int log;		/* whether or not it goes in the command log */
};
static struct control_command_rec control_commands[] = {
	{ "load", "l", cc_load, 1 },
	{ "save", "sa", cc_save, 1 },	/* XXX: cc_write() should be finished */
	{ "copy", "c", cc_copy, 1 },
	{ "filter", "f", cc_filter, 1 },
	{ "write", "w", cc_write, 0 },
	{ "generate", "gen", cc_generate, 1 },
	{ "source", ".", cc_source, 0 },
	{ "play", "p", cc_play, 0 },
	{ "stop", "s", cc_stop, 0 },
	{ "add", "a", cc_add, 1 },
	{ "edit", "e", cc_edit, 1 },
	{ "drop", "d", cc_drop, 1 },
	{ "setlength", "len", cc_setlength, 1 },
	{ "set", NULL, cc_set, 0 },
	{ "info", "i", cc_info, 0 },
	{ "nop", NULL, cc_nop, 0 },
	{ "quit", "q", cc_quit, 0 }
};

void
control_command(char *orig_line)
{
	char *argv[CONTROL_MAX_ARGS];
	int argc = 0;
	char quotechar;
	int i;
	char line_buf[CMD_BUFLEN];
	char *line;

	strncpy(line_buf, orig_line, CMD_BUFLEN);
	line_buf[CMD_BUFLEN-1] = 0;
	line = line_buf;

	memset(argv,0,sizeof argv);

	while (*line) {
		while (isspace(*line))
			line++;

		if (!*line)
			break;

		if (*line == '"' || *line == '\'') {
			quotechar = *line;
			line++;
			if (argc == CONTROL_MAX_ARGS) {
				fprintf(stderr,"line too int");
				break;
			}
			argv[argc++] = line;
			while (*line && *line != quotechar) {
				if (*line == '\\')
					line++;
				if (*line)
					line++;
			}
			if (*line)
				*line++ = 0;
		} else if (*line == '#') {
			/* comment, skip to the end of the line */
			while (*line)
				line++;
		} else {
			if (argc == CONTROL_MAX_ARGS) {
				fprintf(stderr,"line too int");
				break;
			}
			argv[argc++] = line;
			while (*line && !isspace(*line)) {
				if (*line == '\\')
					line++;
				if (*line)
					line++;
			}
			if (*line)
				*line++ = 0;
		}
	}

	/* blank line or comment... */
	if (argc == 0)
		return;

	for (i = 0; i < (sizeof control_commands/sizeof control_commands[0]);
	     i++) {
		if (!strcmp(argv[0],control_commands[i].name) ||
		    (control_commands[i].short_name &&
		     !strcmp(argv[0],control_commands[i].short_name))) {
			/* log it */
			if (!control_commands[i].log) {
				log_write("# ");
			}
			log_write(orig_line);
			log_write("\n");

			/* then call the function to actually do it */
			control_commands[i].func(argc,argv);
			break;
		}
	}
	if (i == (sizeof control_commands/sizeof control_commands[0])) {
		cmd_write("%s: unknown command\n",argv[0]);
	}
}



/* the commands... */

/* load <filename> [identifier] */
static void
cc_load(int argc, char **argv)
{
	struct waveform_rec *wav;

	if (argc != 2 && argc != 3) {
		cmd_write("usage: load <filename> [identifier]\n");
		return;
	}

	wav = wav_read(argv[1]);
	if (argc == 3 && wav) {
		strncpy(wav->name, argv[2], 63);
		wav->name[63] = 0;
	}
	dispwav_refresh();
}

/* save <filename> <identifier> */
static void
cc_save(int argc, char **argv)
{
	struct waveform_rec *wav;

	if (argc != 3) {
		cmd_write("usage: save <filename> <identifier>\n");
		return;
	}

	wav = wav_find(argv[2]);
	if (wav) {
		wav_write(wav, argv[1]);
	}
}

/* copy <identifier> [@<offset>] {tracks [<timespec>]|track <trackspec> [left|right]|<sourcespec> [*<multiplier>]} */
/* destination first! offset is an offset into destination
 * (most useful if destination already exists) */
static void
cc_copy(int argc, char **argv)
{
	int i;
	unsigned offset = 0;
	struct waveform_rec *dest;
	struct waveform_rec *src;
	unsigned start;
	unsigned stop;
	double multiplier = 1.0;

	if (argc < 2) {
		cmd_write("usage: copy <destination-name> [@<offset>] {tracks [<timespec>]|track <trackspec>|<sourcespec> [*<multiplier>]}\n");
		return;
	}
	i = 2;
	if (argv[i][0] == '@') {
		offset = read_time(argv[i]+1);
		i++;
	} else {
		/* no offset spec, so this should probably be a brand new
		 * waveform, so delete the old one */
		if ((dest = wav_find(argv[1])))
			wav_drop(dest);
	}
	dest = wav_find(argv[1]);
	if (!dest) {
		dest = wav_make(argv[1]);
	}

	if (i < argc && !strcmp(argv[i],"track")) {
		struct track_rec *track;
		int chan = 0;
		i++;
		i = scan_trackspec(argc,argv,i,&track,&start,&stop);
		if (i <= 0) {
			cmd_write("bad track selector\n");
			return;
		}
		if (i < argc) {
			if (!strcmp(argv[i],"left")) {
				chan = 0;
				i++;
			} else if (!strcmp(argv[i],"right")) {
				chan = 1;
				i++;
			}
		}
		render_track(track, start, stop, dest, chan);
	} else if (i < argc && !strcmp(argv[i],"tracks")) {
		int chan = 0;
		i++;
		i = scan_timespec(argc,argv,i,&start,&stop);
		if (i < argc) {
			if (!strcmp(argv[i],"left")) {
				chan = 0;
				i++;
			} else if (!strcmp(argv[i],"right")) {
				chan = 1;
				i++;
			}
		}
		render_track(NULL, start, stop, dest, chan);
	} else {
		i = scan_sourcespec(argc,argv,i,&src,&start,&stop);
		if (i == -1) {
			cmd_write("invalid source\n");
			return;
		}
		if (i < argc && argv[i][0] == '*') {
			multiplier = strtod(argv[i]+1,NULL);
			i++;
		}
		wav_merge(dest, offset, src, start, stop, multiplier);
	}
	dispwav_refresh();
	if (i < argc) {
		cmd_write("too many arguments, ignoring at %s\n",argv[i]);
	}
}

/* filter <identifier> <filtername> [filter args ...] */
static void
cc_filter(int argc, char **argv)
{
	/* XXX */
	dispwav_refresh();
}

/* write <sourcespec> <filename> */
static void
cc_write(int argc, char **argv)
{
	/* XXX */
}

/* generate <identifier> <generatorname> [generator args ...] */
static void
cc_generate(int argc, char **argv)
{
	/* XXX */
	dispwav_refresh();
}

/* source <filename> */
static void
cc_source(int argc, char **argv)
{
	FILE *f;
	char buf[CMD_BUFLEN];

	if (argc != 2) {
		cmd_write("usage: source <filename>\n");
		return;
	}
	f = fopen(argv[1],"r");
	if (!f) {
		cmd_write("%s: %s\n",argv[1],strerror(errno));
		return;
	}

	cmd_write("reading %s\n",argv[1]);

	while (fgets(buf,CMD_BUFLEN,f)) {
		buf[strcspn(buf,"\r\n")] = 0;
		control_command(buf);
	}
	fclose(f);
}

/* play {tracks [<timespec>]|track <trackspec>|<sourcespec>} [loop] */
static void
cc_play(int argc, char **argv)
{
	int i = 1;
	unsigned start, stop;

	snd_stop();

	if (argc <= i || !strcmp(argv[i],"tracks")) {
		i++;
		i = scan_timespec(argc,argv,i,&start,&stop);
		snd_track(NULL, start, stop);
	} else if (!strcmp(argv[i],"track")) {
		struct track_rec *track;
		i++;
		i = scan_trackspec(argc,argv,i,&track,&start,&stop);
		if (i <= 0) {
			cmd_write("usage: play {tracks [<timespec>]|track <trackspec>|<sourcespec>} [loop]\n");
			return;
		}
		snd_track(track, start, stop);
	} else {
		struct waveform_rec *wav;
		i = scan_sourcespec(argc,argv,i,&wav,&start,&stop);
		if (i <= 0) {
			cmd_write("usage: play {tracks [<timespec>]|track <trackspec>|<sourcespec>} [loop]\n");
			return;
		}
		snd_wav(wav, start, stop);
	}

	if (i < argc && !strcmp(argv[i],"loop")) {
		snd_loop = 1;
		i++;
	}

	if (i < argc) {
		cmd_write("too many arguments, ignoring at %s\n",argv[i]);
	}
}

/* stop */
static void
cc_stop(int argc, char **argv)
{
	snd_stop();
}


static struct event_rec last_evt;

/* add <eventspec> */
static void
cc_add(int argc, char **argv)
{
	static int event_counter = 1;
	struct event_rec *e;
	int i = 1;

	e = (struct event_rec *)malloc(sizeof *e);

	*e = last_evt;

	i = scan_eventspec(argc,argv,i,e);

	if (i != -1) {
		e->handle = event_counter++;
		track_link(e);
		last_evt = *e;
		disptrack_refresh();
	} else {
		free(e);
		cmd_write("usage: add <eventspec>\n");
	}
}

/* edit <eventno> <eventspec> */
static void
cc_edit(int argc, char **argv)
{
	int i = 1;
	struct event_rec e, *ep = NULL;

	if (argc < 2) {
		cmd_write("usage: edit <eventno> <eventspec>\n");
		return;
	}
	ep = track_event(argv[i]);
	i++;
	if (!ep) {
		cmd_write("event not found\n");
		return;
	}
	e = *ep;
	
	i = scan_eventspec(argc, argv, i, &e);

	if (i != -1) {
		last_evt = e;
		if (e.offset != ep->offset || e.parent != ep->parent) {
			track_unlink(ep);
			*ep = e;
			track_link(ep);
		} else {
			*ep = e;
		}
		disptrack_refresh();
	} else {
		last_evt = *ep;
	}
}

/* drop {event <eventno>|track <trackno>|<identifier>} */
static void
cc_drop(int argc, char **argv)
{
	struct waveform_rec *wav;

	if (argc == 3 && !strcmp(argv[1], "event")) {
		struct event_rec *evt;
		if ((evt = track_event(argv[2]))) {
			track_unlink(evt);
			free(evt);
			disptrack_refresh();
		} else {
			cmd_write("no such event!\n");
		}
	} else if (argc == 3 && !strcmp(argv[1], "track")) {
		track_drop(argv[2]);
		disptrack_refresh();
	} else if (argc == 2 && (wav = wav_find(argv[1]))) {
		wav_drop(wav);
		dispwav_refresh();
	} else {
		cmd_write("usage: drop event <eventno>\n");
		cmd_write("or   : drop track <trackno>\n");
		cmd_write("or   : drop <identifier>\n");
	}
}

/* setlength <identifier> <duration> */
static void
cc_setlength(int argc, char **argv)
{
	struct waveform_rec *wav;
	unsigned len;

	if (argc != 3 || !(wav = wav_find(argv[1]))) {
		cmd_write("usage: setlength <identifier> <duration>\n");
		return;
	}
	len = read_time(argv[2]);
	if (len > wav->len) {
		wav_lengthen(wav, len);
	}
	if (len < wav->len) {
		wav->len = len;
	}
	dispwav_refresh();
}

static struct {
	char *name;
	int *var;
} settable_vars[] = {
	{ "dwscale", &dispwav_scale },
	{ "dtscale", &disptrack_scale },
};

#define NUM_VARS (sizeof settable_vars / sizeof settable_vars[0])

/* set [<name> [<val>]] */
static void
cc_set(int argc, char **argv)
{
	int i;
	if (argc == 1) {
		for (i = 0; i < NUM_VARS; i++) {
			cmd_write("set %s %d\n",settable_vars[i].name,*settable_vars[i].var);
		}
	} else if (argc == 2 || argc == 3) {
		for (i = 0; i < NUM_VARS; i++) {
			if (!strcmp(settable_vars[i].name, argv[1]))
				break;
		}
		if (i == NUM_VARS) {
			cmd_write("set: no such variable '%s'\n",argv[1]);
			return;
		}
		if (argc == 3) {
			*settable_vars[i].var = read_time(argv[2]);
			dispwav_refresh();
			disptrack_refresh();
		} else {
			cmd_write("set %s %d\n",settable_vars[i].name,*settable_vars[i].var);
		}
	} else {
		cmd_write("usage: set [<name> [<val>]]\n");
	}
}

static void
cc_info(int argc, char **argv)
{
	if (argc < 2) {
		cmd_write("info wavs         - waveform summaries\n"
			  "info wav <name>   - waveform details\n");
		return;
	}
	if (!strcmp(argv[1],"wavs")) {
		struct waveform_rec *w;
		for (w = wavs; w; w = w->next) {
			int secs = (w->len + 22050) / 44100;
			int mb = w->buflen/(1024*1024);
			int mins = secs / 60;
			secs %= 60;
			cmd_write("   %02d:%02d %s (%dMB)\n", mins, secs, w->name, mb);
		}
	} else if (!strcmp(argv[1],"wav")) {
		struct waveform_rec *w;
		if (argc < 3 || !(w = wav_find(argv[2]))) {
			cmd_write("no waveform\n");
			return;
		}
		/* XXX: peak, rms, duration   .... ? */
	} else {
		cmd_write("unknown info topic\n");
	}
}

static void
cc_nop(int argc, char **argv)
{
}

static void
cc_quit(int argc, char **argv)
{
	do_cleanup();
}

/* some helpers for the commands: */

static unsigned
read_time(char *s)
{
	unsigned i = 0;

	while (isdigit(*s)) {
		i = (i*10) + (*s-'0');
		s++;
	}
	i *= 1000;
	if (*s == '.') {
		if (isdigit(s[1])) {
			i += (s[1]-'0')*100;
			if (isdigit(s[2])) {
				i += (s[2]-'0')*10;
				if (isdigit(s[3])) {
					i += s[3]-'0';
				}
			}
		}
	}
	return i;
}

/* either "left,right" or just "vol" - the volume is from 0.0 to 1.0,
 * though higher values will result in uh amplification */
static void
read_vol(char *s, unsigned *lvol, unsigned *rvol)
{
	double d;
	d = strtod(s,&s);
	*lvol = (d * 32768.0);
	if (*s == ',') {
		s++;
		d = strtod(s,&s);
		*rvol = (d * 32768.0);
	} else {
		*rvol = *lvol;
	}
}

/* for now just read an integer which is in ratio to 32768 */
static unsigned
read_freq(char *s)
{
	return atoi(s);	/* XXX - add freq table i guess */
}

/* timespec: [<starttime>] [+<duration>]|[:<stoptime>] */
static int scan_timespec(int argc, char **argv, int ind,
		unsigned *start, unsigned *stop)
{
	*start = 0;
	*stop = -1;

	if ((ind < argc) &&
	    (isdigit(argv[ind][0]) || argv[ind][0] == '.')) {
		*start = read_time(argv[ind]);
		ind++;
	}
	if ((ind < argc) &&
	    (argv[ind][0] == '+')) {
		*stop = *start + read_time(argv[ind]+1);
		ind++;
	} else if ((ind < argc) &&
		   (argv[ind][0] == ':')) {
		*stop = read_time(argv[ind]+1);
		ind++;
	}

	return ind;
}

/* sourcespec: <identifier> [<timespec>] */
static int
scan_sourcespec(int argc, char **argv, int ind, struct waveform_rec **wav,
		unsigned *start, unsigned *stop)
{
	struct waveform_rec *t;
	int truncated = 0;

	if ((ind < argc) && (t = wav_find(argv[ind]))) {
		*wav = t;
		ind++;
	} else {
		cmd_write("no source specified\n");
		return -1;
	}

	ind = scan_timespec(argc,argv,ind,start,stop);

	if (*stop == -1)	/* play to end */
		*stop = t->len - 1;

	if (*start >= (*wav)->len) {
		*start = (*wav)->len - 1;
		truncated = 1;
	}
	if (*stop >= (*wav)->len) {
		*stop = (*wav)->len - 1;
		truncated = 1;
	}
	if (truncated) {
		cmd_write("warning: invalid source range reduced to %d:%d (%d)\n",*start,*stop,(*stop-*start));
	}

	*start = wav_find_zero(*wav, *start);
	*stop = wav_find_zero(*wav, *stop);

	return ind;
}

/* trackspec: [<trackno>] [<timespec>] */
static int
scan_trackspec(int argc, char **argv, int ind, struct track_rec **track,
	       unsigned *start, unsigned *stop)
{
	struct track_rec *t;

	if ((ind < argc) && (t = track_find(argv[ind]))) {
		*track = t;
		ind++;
	} else {
		cmd_write("no track specified\n");
		return -1;
	}

	ind = scan_timespec(argc,argv,ind,start,stop);

	return ind;
}

/* eventspec:
 *   [tr <trackno>]
 *   [at <offset>]
 *   [wav <identifier>]
 *     [freq <freq>]
 *     [vol <vol>]
 *     [loop <duration>]
 *   [stop]
 */
static int
scan_eventspec(int argc, char **argv, int ind, struct event_rec *e)
{
	while (ind < argc) {
		if (!strcmp(argv[ind], "tr")) {
			struct track_rec *tr;
			ind++;
			if (ind == argc || !(tr = track_find(argv[ind]))) {
				cmd_write("no track\n");
				return -1;
			}
			ind++;
			e->parent = tr;
		} else if (!strcmp(argv[ind], "at")) {
			ind++;
			if (ind == argc) {
				cmd_write("no time\n");
				return -1;
			}
			e->offset = read_time(argv[ind]);
			ind++;
		} else if (!strcmp(argv[ind], "wav")) {
			struct waveform_rec *w;
			ind++;
			if (ind == argc || !(w = wav_find(argv[ind]))) {
				cmd_write("no wav\n");
				return -1;
			}
			ind++;
			if (e->kind != ek_sample) {
				e->kind = ek_sample;
				e->u.sample.freq = 32768;
				e->u.sample.lvol = 32768;
				e->u.sample.rvol = 32768;
			}
			e->u.sample.wav = w;
		} else if (!strcmp(argv[ind], "vol")) {
			ind++;
			if (ind == argc) {
				cmd_write("no vol\n");
				return -1;
			}
			if (e->kind == ek_sample) {
				read_vol(argv[ind],
					 &e->u.sample.lvol, &e->u.sample.rvol);
			} else {
				cmd_write("vol doesn't apply to this event\n");
				return -1;
			}
			ind++;
		} else if (!strcmp(argv[ind], "freq")) {
			ind++;
			if (ind == argc) {
				cmd_write("no freq\n");
				return -1;
			}
			if (e->kind == ek_sample) {
				unsigned freq = read_freq(argv[ind]);
				if (freq) {
					e->u.sample.freq = freq;
				} else {
					cmd_write("invalid freq\n");
					return -1;
				}
			} else {
				cmd_write("freq doesn't apply to this event\n");
				return -1;
			}
			ind++;
		} else if (!strcmp(argv[ind], "loop")) {
			ind++;
			if (ind == argc) {
				cmd_write("no loop len\n");
				return -1;
			}
			if (e->kind == ek_sample) {
				unsigned loop = read_time(argv[ind]);
				e->u.sample.loop_len = loop;
			} else {
				cmd_write("loop doesn't apply to this event\n");
				return -1;
			}
			ind++;
		} else if (!strcmp(argv[ind], "stop")) {
			ind++;
			e->kind = ek_stop;
		} else {
			cmd_write("unknown argument '%s'\n",argv[ind]);
			return -1;
		}
	}
	if (!e->parent) {
		cmd_write("no track selected!\n");
		return -1;
	}
	switch (e->kind) {
		case ek_sample:
			if (!e->u.sample.wav) {
				cmd_write("no waveform selected!\n");
				return -1;
			}
			break;
		case ek_stop:
			break;
		default:
			cmd_write("no event kind selected!\n");
			return -1;
	}
	return ind;
}

/* eventspec takes an event record and fills in the specified values..  for add, the event record is the previous one referenced, and for edit, the event record is the one to edit.  if adding the first one and incomplete spec, well, wtf */

