#ifndef GGA_H
#define GGA_H

	/* used by cmd */
#define HIST_MAX 100000
#define HIST_FILE ".gga_history"
#define CMD_BUFLEN 10240

	/* used by control */
#define STARTUP_FILE ".gga_startup"
#define LOG_FILE ".gga_log"
#define CONTROL_MAX_ARGS 128

	/* used by snd */
#define SND_FILE "/dev/dsp"	/* should be selectable by cmdline */
#define FRAG_LEN 12		/* 2^12 => 4096 bytes per fragment */
#define ZERO_CROSS_SIZE 500	/* how far to look for zero crossings */

	/* used by x */
#define X_BIGFONT "-greg-vga-*-24-*"
#define X_SMALLFONT "-greg-vga-*-16-*"


/**** waveform structures ****/
struct waveform_rec {
	struct waveform_rec *next;
	char name[64];
	signed short *data;
	unsigned len;
	unsigned buflen;
};

extern struct waveform_rec *wavs;



/**** tracker structures ****/
struct track_rec {
	struct track_rec *next;
	struct event_rec *events;	/* ordered by time */
};

struct event_rec {
	struct event_rec *next;
	int handle;			/* unique identifier int */
	struct track_rec *parent;	/* the track it beints to */
	unsigned offset;
	enum {
		ek_unknown,
		ek_sample,
		ek_stop
	} kind;
	union {
		struct {
			struct waveform_rec *wav;
			unsigned freq;
			unsigned lvol,rvol;
			unsigned loop_len;
		} sample;
	} u;
};

extern struct track_rec *tracks;



/**** cmd.c ****/

extern int cmd_fd;		/* the fd to select on for input */

int cmd_init(void);		/* initialize cmd thread */
void cmd_write(char *fmt,...);	/* write this to the user */
void cmd_read(void);		/* reads a line when input is available */
void cmd_cleanup(void);



/**** control.c ****/

int control_init(void);		/* reads&executes the 'startup' file */
int control_loop(void);		/* the main select() loop etc */
void control_command(char *orig_line);	/* process a single input line */



/**** disptrack.c ****/

extern int disptrack_scale;	/* number of samples shown at once */

int disptrack_init(void);
void disptrack_refresh(void);
void disptrack_location(int x,int y);



/**** dispwav.c ****/

extern int dispwav_scale;	/* number of samples shown at once */

int dispwav_init(void);
void dispwav_refresh(void);
void dispwav_location(int x,int y);



/**** snd.c ****/

extern int snd_fd;			/* output sound fd */
extern int snd_ready;			/* snd subsystem has output for it */
extern int snd_loop;			/* currently playing sound repeats.. */

int snd_init(void);			/* open snd_fd.. */
void snd_doit(void);			/* write to snd_fd */
void snd_wav(struct waveform_rec *wav, unsigned start, unsigned stop);
void snd_track(struct track_rec *track, unsigned start, unsigned stop);
void render_track(struct track_rec *track, unsigned start, unsigned stop,
        struct waveform_rec *wav, int chan); /* render track to waveform */
void snd_stop();			/* stop all output.. */



/**** x.c ****/

extern int x_fd;			/* the connection to the server */

int x_init(int argc,char **argv);	/* initialize X connection */
void x_flush(void);			/* prepare to wait for input */
void x_event(void);			/* we got activity on x_fd */



/**** wav.c ****/

struct waveform_rec *wav_read(char *fn);	/* load a waveform */
void wav_write(struct waveform_rec *wav, char *fn); /* save a waveform */
struct waveform_rec *wav_find(char *s);		/* select a waveform */
void wav_drop(struct waveform_rec *wav);	/* delete a waveform */
void wav_merge(struct waveform_rec *dest, unsigned offset,
          struct waveform_rec *src, unsigned start, unsigned stop,
          double mult);				/* merge waveforms */
struct waveform_rec *wav_make(char *ident);	/* allocate a new one */
void wav_lengthen(struct waveform_rec *wav, unsigned len);
unsigned wav_find_zero(struct waveform_rec *wav, unsigned index);
						/* find a zero crossing */



/**** track.c ****/

struct track_rec *track_find(char *name);
void track_drop(char *name);
void track_link(struct event_rec *e);
void track_unlink(struct event_rec *e);
struct event_rec *track_event(char *name);



/**** gga.c ****/

void do_cleanup(void);				/* exit the program */

#endif /* GGA_H */
