#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <limits.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>
#include "gga.h"

int snd_fd = -1;			/* output sound fd */
int snd_ready = 0;			/* snd subsystem has output for it */
int snd_loop = 0;			/* currently playing sound repeats.. */

static enum { pl_none, pl_wav, pl_track, pl_tracks } curr_playing = pl_none;
static struct waveform_rec *curr_wav = NULL;
static struct track_rec *curr_track = NULL;
static unsigned curr_start,curr_stop;
static unsigned curr_index;
static int track_done;			/* any track that has remaining events
					 * should clear this */
static int track_overflowed = 0;

struct sample_rec {
	struct sample_rec *next;
	struct waveform_rec *wav;
	struct track_rec *tr;		/* the track it belongs to */
	unsigned start_index;
	unsigned ofs;
	unsigned lvol,rvol;		/* 32786 = full */
	unsigned remainder;
	unsigned freq;			/* 32768 = default */
	unsigned loop;			/* time from start_index for repeats */
};
static struct sample_rec *samples = NULL;	/* currently playing samples */

int
snd_init(void)
{
	int i;

	snd_fd = open(SND_FILE, O_WRONLY);
	if (snd_fd < 0) {
		perror(SND_FILE);
		return -1;
	}
	i = 0x7ffe0000 | FRAG_LEN;
	ioctl(snd_fd,SNDCTL_DSP_SETFRAGMENT,&i);
	if (i != (0x7ffe0000 | FRAG_LEN)) {
		perror("ioctl(SNDCTL_DSP_SETFRAGMENT,0x7ffe0000");
		return -1;
	}
	i = AFMT_S16_LE;		/* PORTABILITY -> _BE for big endian */
	ioctl(snd_fd,SNDCTL_DSP_SETFMT,&i);
	if (i != AFMT_S16_LE) {
		perror("ioctl(SNDCTL_DSP_SETFMT,AFMT_S16_LE");
		return -1;
	}
	i = 1;
	ioctl(snd_fd,SNDCTL_DSP_STEREO,&i);
	if (i != 1) {
		perror("ioctl(SNDCTL_DSP_STEREO,1");
		return -1;
	}
	i = 44100;
	ioctl(snd_fd,SNDCTL_DSP_SPEED,&i);
	if (i < 41895 || i > 46305) {	/* 5% tolerances */
		perror("ioctl(SNDCTL_DSP_SPEED,44100");
		return -1;
	}

	snd_stop();
	return 0;
}

static signed short sndbuf[(1<<FRAG_LEN)/2];

static void
doit_wav(void)
{
	int i;
	for (i = 0; i < sizeof sndbuf/4; i++) {
		if (curr_index >= curr_stop || curr_index >= curr_wav->len) {
			if (!snd_loop || curr_index == curr_start) {
				snd_stop();
				return;
			}
			curr_index = curr_start;
		}
		sndbuf[i*2+0] += curr_wav->data[curr_index];
		sndbuf[i*2+1] += curr_wav->data[curr_index];
		curr_index++;
	}
	/* XXX: maybe play should let you test different volumes? */
}

static void
evt_stop(struct track_rec *t)
{
	struct sample_rec *s, **sp;
	for (sp = &samples; *sp; sp = &((*sp)->next)) {
		while (*sp && (!t || (*sp)->tr == t)) {
			s = *sp;
			*sp = s->next;
			free(s);
		}
		if (!*sp)
			return;
	}
}

static void
doit_track(struct track_rec *t)
{
	int i;
	struct event_rec *e;
	struct sample_rec *s, *prev;

		/* skip finished events */
	for (e = t->events; e && e->offset < curr_index; e = e->next)
		;

	for (i = 0; i < sizeof sndbuf/4; i++) {
		if (curr_index >= curr_stop) {
			if (!snd_loop)
				return;
			curr_index = curr_start;
				/* reposition in the event stream... */
			for (e = t->events; e && e->offset < curr_index;
			     e = e->next)
				;
		}
		while (e && e->offset == curr_index) {
			switch (e->kind) {
				case ek_sample:
					s = (struct sample_rec *)malloc(sizeof *s);
					memset(s,0,sizeof *s);
					s->next = samples;
					s->start_index = curr_index;
					s->wav = e->u.sample.wav;
					s->tr = t;
					s->lvol = e->u.sample.lvol;
					s->rvol = e->u.sample.rvol;
					s->freq = e->u.sample.freq;
					s->loop = e->u.sample.loop_len;
					samples = s;
					break;
				case ek_stop:
					evt_stop(t);
					break;
				default: break;
			}
			e = e->next;
		}
		prev = NULL;
		for (s = samples; s; s = s->next) {
			signed left,right;
retry:
			if (s->tr != t) {
				prev = s;
				continue;
			}
			s->remainder += s->freq;
			s->ofs += s->remainder >> 15;
			s->remainder &= 0x7fff;
			if (s->ofs >= s->wav->len &&
			    s->loop == 0) {
				/* end of this sample... */
				struct sample_rec *t = s;
				s = s->next;
				if (prev) {
					prev->next = s;
				} else {
					samples = s;
				}
				free(t);
				if (!s)
					break;
				goto retry;
			} else if (s->ofs >= s->wav->len &&
				   s->loop + s->start_index > curr_index) {
				/* waiting for loop, do nothing */
				continue;
			} else if (s->loop &&
				   s->loop + s->start_index == curr_index) {
				s->ofs = 0;
				s->remainder = 0;
				s->start_index = curr_index;
			}
			left  = s->wav->data[s->ofs];
			right = s->wav->data[s->ofs];
			left  = (left  * (signed)s->lvol) >> 15;
			right = (right * (signed)s->rvol) >> 15;
			left  += (signed)sndbuf[i*2+0];
			right += (signed)sndbuf[i*2+1];
			if (left  < SHRT_MIN || left  > SHRT_MAX ||
			    right < SHRT_MIN || right > SHRT_MAX) {
				track_overflowed = 1; /* XXX - display this */
			}
			sndbuf[i*2+0] += left;
			sndbuf[i*2+1] += right;

			prev = s;
		}
		curr_index++;
	}
	if (e)
		track_done = 0;
}

void snd_doit(void)
{
	memset(sndbuf, 0, sizeof sndbuf);

	switch (curr_playing) {
		case pl_wav:
			doit_wav();
			break;
		case pl_track:
			track_done = 1;
			doit_track(curr_track);
			if (track_done &&
			    (!samples || curr_index >= curr_stop)) {
				if (snd_loop)
					curr_index = curr_start;
				else
					snd_stop();
			}
			break;
		case pl_tracks: {
			struct track_rec *t;
			unsigned idx = curr_index;
			track_done = 1;
			for (t = tracks; t; t = t->next) {
				curr_index = idx;
				doit_track(t);
			}
			if (track_done &&
			    (!samples || curr_index >= curr_stop)) {
				if (snd_loop)
					curr_index = curr_start;
				else
					snd_stop();
			}
		}	break;
		default:
			snd_stop();
	}

	write(snd_fd, sndbuf, sizeof sndbuf);
}

void
snd_wav(struct waveform_rec *wav, unsigned start, unsigned stop)
{
	curr_playing = pl_wav;
	curr_wav = wav;
	curr_start = start;
	curr_stop = stop;
	curr_index = start;
	snd_ready = 1;
}


/* call with track=NULL to play all tracks */
void
snd_track(struct track_rec *track, unsigned start, unsigned stop)
{
	if (track) {
		curr_playing = pl_track;
		curr_track = track;
	} else {
		curr_playing = pl_tracks;
	}
	curr_start = start;
	curr_stop = stop;
	curr_index = start;
	snd_ready = 1;
}


/* renders either the left channel (chan=0) or right channel (chan=1) of 
 * the specified track (or all tracks if track=NULL), stores it into
 * wav, overwriting whatever may be there */
void
render_track(struct track_rec *track, unsigned start, unsigned stop,
             struct waveform_rec *wav, int chan)
{
	struct track_rec *tr;
	unsigned dest_ofs;
	unsigned dest_len;
	unsigned i;

	snd_stop();

	curr_start = start;
	curr_stop = stop;
	curr_index = start;

	for (;;) {
		unsigned idx = curr_index;
		memset(sndbuf, 0, sizeof sndbuf);
		track_done = 1;
		for (tr = (track ? track : tracks);
		     tr;
		     tr = (track ? NULL : tr->next)) {
			curr_index = idx;
			doit_track(tr);
		}

		dest_ofs = idx - curr_start;
		dest_len = curr_stop - idx;
		if (dest_len > sizeof sndbuf/4)
			dest_len = sizeof sndbuf/4;
		wav_lengthen(wav, dest_ofs+dest_len);
		for (i = 0; i < dest_len; i++) {
			wav->data[i+dest_ofs] = chan
					? sndbuf[i*2+1]
					: sndbuf[i*2+0];
		}

		if (track_done &&
		    (!samples || curr_index >= curr_stop)) {
			break;
		}
	}

	snd_stop();
}

void
snd_stop()
{
	struct sample_rec *s,*sn;

	for (s = samples; s; s = sn) {
		sn = s->next;
		free(s);
	}
	samples = NULL;
	curr_playing = pl_none;
	snd_ready = 0;
	snd_loop = 0;
}

