#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <assert.h>
#include <sys/mman.h>
#include "gga.h"

static void *wav_alloc(int len);
static void wav_free(void *ptr, int len);



struct waveform_rec *
wav_read(char *fn)
{
	struct waveform_rec *wav;
	FILE *f;
	int len;
	int file_len;
	int suffix = 0;
	char *fn2;

	if (!(f = fopen(fn,"r"))) {
		perror(fn);
		return NULL;
	}

	wav = (struct waveform_rec *)malloc(sizeof *wav);

	fn2 = rindex(fn, '/');	/* skip directory part */
	if (fn2)
		fn2 += 1;
	else
		fn2 = fn;
	strncpy(wav->name, fn2, 50);
	wav->name[50] = 0;
	len = strcspn(wav->name,".");
	wav->name[len] = 0;

	while (wav_find(wav->name)) {
		sprintf(wav->name+len, "%d", suffix++);
	}

	fseek(f, 0, SEEK_END);
	file_len = ftell(f);
	fseek(f, 0, SEEK_SET);

	wav->len = file_len / 2;
	wav->buflen = (file_len + 0xFFFFF) & (~0xFFFFF); /* round up to MB */
	wav->data = wav_alloc(wav->buflen);
	fread(wav->data, file_len, 1, f);

	wav->next = wavs;
	wavs = wav;

	fclose(f);

	cmd_write("loaded %s into '%s'\n",fn,wav->name);

	return wav;
}

void
wav_write(struct waveform_rec *wav, char *fn)
{
	FILE *f;

	if (!(f = fopen(fn,"w"))) {
		perror(fn);
		return;
	}
	fwrite(wav->data, wav->len*2, 1, f);
	fclose(f);
}

struct waveform_rec *
wav_find(char *s)
{
	struct waveform_rec *w;
	for (w = wavs; w; w = w->next) {
		if (!strcmp(s,w->name))
			return w;
	}
	return NULL;
}

void
wav_drop(struct waveform_rec *wav)
{
	struct waveform_rec *t;

	/* get it out of the list */
	if (wavs == wav) {
		wavs = wavs->next;
	} else {
		for (t = wavs; t && t->next != wav; t = t->next)
			;
		assert(t);	/* it better be in the list... */
		t->next = wav->next;
	}

	/* free the memory */
	wav_free(wav->data, wav->buflen);
	free(wav);
}

void
wav_merge(struct waveform_rec *dest, unsigned offset,
          struct waveform_rec *src, unsigned start, unsigned stop,
          double mult)
{
	unsigned len = stop-start;
	signed short *d;
	signed short *s;
	signed int val;
	int overflow = 0;
	int use_mult = (mult != 1.0);

	wav_lengthen(dest, offset+len);

	d = dest->data + offset;
	s = src->data + start;
	
	while (len) {
		val = *d;
		if (use_mult) {
			val += (mult * ((double)*s));
		} else {
			val += *s;
		}
		if (val < SHRT_MIN || val > SHRT_MAX)
			overflow++;
		*d = val;
		len--;
		s++;
		d++;
	}
	if (overflow) {
		cmd_write("overflow on approximately %d samples!\n",overflow);
	}
}


struct waveform_rec *
wav_make(char *ident)
{
	struct waveform_rec *wav;

	wav = (struct waveform_rec *)malloc(sizeof *wav);
	strncpy(wav->name, ident, 63);
	wav->name[63] = 0;
	wav->data = NULL;
	wav->len = wav->buflen = 0;
	wav->next = wavs;
	wavs = wav;

	return wav;
}

void
wav_lengthen(struct waveform_rec *wav, unsigned len)
{
	void *newp;
	unsigned newlen;

	if (wav->len >= len)	/* do not shorten */
		return;
	if (wav->buflen >= len*2) {
		wav->len = len;
		return;
	}
	newlen = ((len*2) + 0xFFFFF) & (~0xFFFFF);
	newp = wav_alloc(newlen);
	memcpy(newp, wav->data, wav->buflen);
	wav->data = newp;
	wav->len = len;
	wav->buflen = newlen;
}


/* searches for a zero crossing within ZERO_CROSS_SIZE samples of this one */
unsigned
wav_find_zero(struct waveform_rec *wav, unsigned index)
{
	unsigned lowest = (index > ZERO_CROSS_SIZE)
			? (index - ZERO_CROSS_SIZE)
			: 0;
	unsigned highest = ((wav->len - index) > ZERO_CROSS_SIZE)
			? (index + ZERO_CROSS_SIZE)
			: wav->len;
	unsigned i;
	int sign;
	unsigned closest = index;
	unsigned dist_found = ZERO_CROSS_SIZE*2;

	if (index < lowest)
		return lowest;
	if (index >= highest)
		return highest;

	sign = (wav->data[index] > 0);

	/* look for one forwards first */
	for (i = index; i < highest; i++) {
		if (sign) {
			if (wav->data[i] <= 0) {
				closest = i;
				dist_found = i - index;
				break;
			}
		} else {
			if (wav->data[i] >= 0) {
				closest = i;
				dist_found = i - index;
				break;
			}
		}
	}

	/* then reverse */
	for (i = index; lowest && i >= lowest; i--) {
		if (sign) {
			if (wav->data[i] <= 0) {
				if (index - i < dist_found)
					closest = i;
				break;
			}
		} else {
			if (wav->data[i] >= 0) {
				if (index - i < dist_found)
					closest = i;
				break;
			}
		}
	}

	/* no candidates found */
	return closest;
}


static void *
wav_alloc(int len)
{
	void *p;
	p = mmap(NULL, len, PROT_READ|PROT_WRITE,
	         MAP_PRIVATE|MAP_ANONYMOUS, -1, 0);
	if (!p) {
		perror("mmap");
		return NULL;
	}
	/* PORTABILITY: should zero it here, but let's trust the OS... */
	return p;
}

static void
wav_free(void *ptr, int len)
{
	if (!ptr || !len)
		return;
	munmap(ptr, len);
}
