#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <getopt.h>
#include <unistd.h>

int mixer_fd=-1,devmask=0,stereodevs=0,recmask=0,recsrc=0;
char device_no=0;
unsigned int verbose=0;

#define DEV_EXIST(dev) ((1<<dev) & devmask)
#define STEREO(dev) ((1<<dev) & stereodevs)

#define NUM_NON_VOL_ARGS 5	/* MAKE SURE TO CHANGE THIS!! */
				/* (if you modify the opts array */

char *devnames[SOUND_MIXER_NRDEVICES]=SOUND_DEVICE_NAMES;
struct option opts[] = { 
	{ "help",0,NULL,'h' },
	{ "display",0,NULL,'d' },
	{ "verbose",0,NULL,'V' },
	{ "recsrc",1,NULL,'R' },
	{ "card",1,NULL,'C' },
	{ "vol",1,NULL,'v' },
	{ "bass",1,NULL,'b' },
	{ "treble",1,NULL,'t' },
	{ "synth",1,NULL,'s' },
	{ "pcm",1,NULL,'p' },
	{ "speaker",1,NULL,'S' },
	{ "line",1,NULL,'l' },
	{ "mic",1,NULL,'m' },
	{ "cd",1,NULL,'c' },
	{ "mix",1,NULL,'x' },
	{ "pcm2",1,NULL,'a' },
	{ "rec",1,NULL,'r' },
	{ "igain",1,NULL,'i' },
	{ "ogain",1,NULL,'o' },
	{ "line1",1,NULL,'1' },
	{ "line2",1,NULL,'2' },
	{ "line3",1,NULL,'3' },
	{ "dig1",1,NULL,'4' },
	{ "dig2",1,NULL,'5' },
	{ "dig3",1,NULL,'6' },
	{ "phin",1,NULL,'I' },
	{ "phout",1,NULL,'O' },
	{ "video",1,NULL,'y' },
	{ "radio",1,NULL,'z' },
	{ "monitor",1,NULL,'M' },
	{ NULL,0,NULL,0 } };

void help(char *argv0);
int display(char *argv0);
int set_recsrc(char *arg);
int set_volume(int dev,char *level);
int get_volume(int dev,int *loft,int *right);
void open_mixer(void);

int main(int argc,char **argv)
{
	int c,i,had_arg=0;
	int option_index=0;
	while ((c=getopt_long(argc,argv,"hdVR:C:v:b:t:s:p:S:l:m:c:x:a:r:i:o:1:2:3:4:5:6:I:O:y:z:M:",opts,&option_index))!=EOF) {
		if (c!='C' && c!='V') had_arg=1;
		switch(c) {
			case '?':
				fprintf(stderr,"%s: unknown argument!\n",argv[0]);
				help(argv[0]);
				exit(-1);
			case ':':
				fprintf(stderr,"%s: argument needs additional argument!\n",argv[0]);
				help(argv[0]);
				exit(-1);
			case 'h':
				help(argv[0]);
				break;
			case 'd':
				display(argv[0]);
				break;
			case 'V':
				verbose++;
				break;
			case 'R':
				set_recsrc(optarg);
				break;
			case 'C':
				device_no=optarg[0];
				if (mixer_fd!=-1) {
					close(mixer_fd);
					mixer_fd=-1;
				}
				break;
			case 'v':
			case 'b':
			case 't':
			case 's':
			case 'p':
			case 'S':
			case 'l':
			case 'm':
			case 'c':
			case 'x':
			case 'a':
			case 'r':
			case 'i':
			case 'o':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case 'I':
			case 'O':
			case 'y':
			case 'z':
			case 'M':
				for (i=0;opts[i].name;i++)
					if (opts[i].val==c)
						break;
				set_volume(i-NUM_NON_VOL_ARGS,optarg);
				break;
			default:
				fprintf(stderr,"unknown option: %c\n",c);
		}
	}
	if (!had_arg)
		display(argv[0]);
	return 0;
}

void help(char *argv0)
{
	fprintf(stderr,"usage: %s [options]\n",argv0);
	fprintf(stderr,"options:\n");
	fprintf(stderr,"-h,--help          get this output\n");
	fprintf(stderr,"-d,--display       display current volume settings\n");
	fprintf(stderr,"-V,--verbose       verbose mode (pretty bargraphs)\n");
	fprintf(stderr,"-C,--card card     select which card the following commands should affect\n");
	fprintf(stderr,"-v,--vol x[,y]     set master volume\n");
	fprintf(stderr,"-b,--bass x[,y]    set bass volume\n");
	fprintf(stderr,"-t,--treble x[,y]  set treble volume\n");
	fprintf(stderr,"-s,--synth x[,y]   set synth (FM/MIDI) volume\n");
	fprintf(stderr,"-p,--pcm x[,y]     set PCM (digital) volume\n");
	fprintf(stderr,"-S,--speaker x[,y] set speaker volume\n");
	fprintf(stderr,"-l,--line x[,y]    set line in volume\n");
	fprintf(stderr,"-m,--mic x[,y]     set microphone volume\n");
	fprintf(stderr,"-c,--cd x[,y]      set CD volume\n");
	fprintf(stderr,"-x,--mix x[,y]     set mixer volume\n");
	fprintf(stderr,"-a,--pcm2 x[,y]    set alternate PCM volume\n");
	fprintf(stderr,"-r,--rec x[,y]     set recording volume level\n");
	fprintf(stderr,"-i,--igain x[,y]   set input gain level\n");
	fprintf(stderr,"-o,--ogain x[,y]   set output gain level\n");
	fprintf(stderr,"-1,--line1 x[,y]   set line1 volume\n");
	fprintf(stderr,"-2,--line2 x[,y]   set line2 volume\n");
	fprintf(stderr,"-3,--line3 x[,y]   set line3 volume\n");
	fprintf(stderr,"-4,--dig1 x[,y]    set digital1 volume\n");
	fprintf(stderr,"-5,--dig2 x[,y]    set digital2 volume\n");
	fprintf(stderr,"-6,--dig3 x[,y]    set digital3 volume\n");
	fprintf(stderr,"-I,--phin x[,y]    set phone in volume\n");
	fprintf(stderr,"-O,--phout x[,y]   set phone out volume\n");
	fprintf(stderr,"-y,--video x[,y]   set video volume\n");
	fprintf(stderr,"-z,--radio x[,y]   set radio volume\n");
	fprintf(stderr,"-M,--monitor x[,y] set monitor volume\n");
	fprintf(stderr,"-R,--recsrc z      set the recording source to z, where z is one of:\n");
	fprintf(stderr,"                   synth,pcm,speaker,line,mic,cd,mix,pcm2,line1,line2,line3\n");
	fprintf(stderr,"                   (most sound cards only support line,mic,cd but the rest\n");
	fprintf(stderr,"                   are included for completeness)\n");
	fprintf(stderr,"\nif y is specified, x is left and y is right.  Otherwise, x is left and right.\n");
}

int display(char *argc0)
{
	int i,l,r,n;

	open_mixer();
	printf("%s",argc0);
	if (device_no) printf(" --card %c",device_no);
	for (i=0;i<SOUND_MIXER_NRDEVICES;i++)
		if (!get_volume(i,&l,&r)) {
			if (l==r)
				printf(" --%s %d",devnames[i],l);
			else
				printf(" --%s %d,%d",devnames[i],l,r);
		}
	if (recsrc)
		for (i=0;i<SOUND_MIXER_NRDEVICES;i++)
			if (recsrc&(1<<i)) {
				printf(" --recsrc %s\n",devnames[i]);
				break;
			}
	printf("\n");
	for (i=0;i<SOUND_MIXER_NRDEVICES;i++)
		if (!get_volume(i,&l,&r)) {
			if (STEREO(i)) {
				printf("%-10s (stereo) %3d,%3d  ",devnames[i],l,r);
				if (verbose) {
					for (n=0;n<50;n++)
						printf("%c",(n*2<=l)?':':'.');
					printf("\n                             ");
					for (n=0;n<50;n++)
						printf("%c",(n*2<=r)?':':'.');
				}
				printf("\n");
			} else {
				printf("%-10s (mono)   %3d      ",devnames[i],l);
				if (verbose) {
					for (n=0;n<50;n++)
						printf("%c",(n*2<=l)?':':'.');
				}
				printf("\n");
			}
		}
	printf("input source(s):");
	for (i=0;i<SOUND_MIXER_NRDEVICES;i++)
		if (recsrc&(1<<i))
			printf(" %s",devnames[i]);
	printf("\n");
	return 0;
}

int set_recsrc(char *arg)
{
	int i;
	if (strlen(arg)==1)
		for (i=NUM_NON_VOL_ARGS;opts[i].name;i++) {
			if (opts[i].val==*arg)
				break;
		}
	else
		for (i=NUM_NON_VOL_ARGS;opts[i].name;i++)
			if (!strcasecmp(opts[i].name,arg))
				break;
	if (!opts[i].name)
		return fprintf(stderr,"source \"%s\" not valid!\n",arg),-1;
	i-=NUM_NON_VOL_ARGS;
	open_mixer();
	if (!(recmask&(1<<i)))
		fprintf(stderr,"source \"%s\" not supported by your sound driver!\n",arg),-1;
	recsrc=1<<i;
	ioctl(mixer_fd,SOUND_MIXER_WRITE_RECSRC,&recsrc);
	return 0;
}

int set_volume(int dev,char *level)
{
	int l,r,t;
	char *s;

	open_mixer();
	if (!DEV_EXIST(dev))
		return fprintf(stderr,"%s device not present!\n",devnames[dev]),-1;
	l=atoi(level);
	if ((s=strchr(level,',')))
		r=atoi(s+1);
	else
		r=l;
	if (l>100)
		l=100;
	if (r>100)
		r=100;
	if (l<0)
		l=0;
	if (r<0)
		r=0;
	t=l|(r<<8);
	if (ioctl(mixer_fd,MIXER_WRITE(dev),&t)==-1)
		return perror("MIXER_WRITE(dev)"),-1;
	return 0;
}

int get_volume(int dev,int *left,int *right)
{
	int n;

	open_mixer();
	if (!DEV_EXIST(dev))
		return -1;
	if (ioctl(mixer_fd,MIXER_READ(dev),&n)==-1)
		exit((perror("MIXER_READ(dev)"),-1));
	*left=n&0xff;
	*right=STEREO(dev)?((n>>8)&0xff):(*left);
	return 0;
}

void open_mixer(void)
{
	char s[100];
	if (mixer_fd!=-1)
		return;
	strcpy(s,"/dev/mixer");
	s[10]=device_no;
	s[11]=0;
	mixer_fd=open(s,O_RDWR);
	if (mixer_fd==-1)
		exit((perror(s),-1));
	if (ioctl(mixer_fd,SOUND_MIXER_READ_DEVMASK,&devmask)==-1)
		exit((perror("ioctl(SOUND_MIXER_READ_DEVMASK)"),-1));
	if (ioctl(mixer_fd,SOUND_MIXER_READ_STEREODEVS,&stereodevs)==-1)
		exit((perror("ioctl(SOUND_MIXER_READ_STEREODEVS)"),-1));
	if (ioctl(mixer_fd,SOUND_MIXER_READ_RECMASK,&recmask)==-1)
		exit((perror("ioctl(SOUND_MIXER_READ_RECMASK)"),-1));
	if (ioctl(mixer_fd,SOUND_MIXER_READ_RECSRC,&recsrc)==-1)
		exit((perror("ioctl(SOUND_MIXER_READ_RECSRC)"),-1));
	return;
}

