#ifndef __INST_H__
#define __INST_H__


#define MAX_TOKENS 128


enum inst_kind {
	IK_LABEL,		/* (0) */
	IK_LITERAL,		/* literal (in args) */
	IK_BYTE,		/* opcode(6), d(1), a(1), f(8) */
	IK_BYTE_FILE,		/* opcode(7), a(1), f(8) */
	IK_BYTE_TO_BYTE,	/* opcode(4), src_f(12), -1(4), dst_f(12) */
	IK_BIT,			/* opcode(4), b(3), a(1), f(8) */
	IK_OP_LIT,		/* opcode(8), lit(8) */
	IK_BRANCH_DIRECT,	/* opcode(8), addrl(8), -1(4), addrh(12) */
	IK_BRANCH_REL_SHORT,	/* opcode(8), ofs(8) */
	IK_BRANCH_REL_LONG,	/* opcode(5), ofs(11) */
	IK_OPONLY,		/* opcode(16) */
	IK_FSR_LIT,             /* opcode(10), fsr(2), lith(4), -1(4), 0(4), litl(8) */
};

enum op_name {
	OP_ADD,			/* add */
	OP_ADDC,		/* add with carry */
	OP_AND,			/* and */
	OP_CLR,			/* clear (set to 0) */
	OP_COMF,		/* complement F (XOR with -1) */
	OP_CMPSEQ,		/* compare, skip if equal */
	OP_CMPSGT,		/* compare, skip if greater */
	OP_CMPSLT,		/* compare, skip if less */
	OP_DECF,		/* decrement F */
	OP_DECFSZ,		/* decrement F, skip if zero */
	OP_DECFSNZ,		/* decrement F, skip if not zero */
	OP_INCF,		/* increment F */
	OP_INCFSZ,		/* increment F, skip if zero */
	OP_INCFSNZ,		/* increment F, skip if not zero */
	OP_IOR,			/* inclusive or */
	OP_MOV,			/* move */
	OP_MUL,			/* 8x8 multiply */
	OP_NEG,			/* negate */
	OP_RLF,			/* rotate F left */
	OP_RLFC,		/* rotate F left through carry */
	OP_RRF,			/* rotate F right */
	OP_RRFC,		/* rotate F right through carry */
	OP_SET,			/* set (set to -1) */
	OP_SUBWB,		/* subtract from W with borrow */
	OP_SUBF,		/* subtract from F */
	OP_SUBFB,		/* subtract from F with borrow */
	OP_SUBLW,		/* subtract from literal */
	OP_SWAPF,		/* swap nibbles of F */
	OP_TSTSZ,		/* test, skip if zero */
	OP_XOR,			/* exclusive or */
	OP_BCLR,		/* bit clear */
	OP_BSET,		/* bit set */
	OP_BTSC,		/* bit test, skip if clear */
	OP_BTSS,		/* bit test, skip if set */
	OP_BTG,			/* bit toggle */
	OP_BC,			/* branch if carry */
	OP_BN,			/* branch if negative */
	OP_BNC,			/* branch if not carry */
	OP_BNN,			/* branch if not negative */
	OP_BNO,			/* branch if not overflow */
	OP_BNZ,			/* branch if not zero */
	OP_BO,			/* branch if overflow */
	OP_BZ,			/* branch if zero */
	OP_GOTO,		/* branch unconditionally */
	OP_CALL,		/* call subroutine */
	OP_FCALL,		/* fast-call subroutine */
	OP_CLRWDT,		/* clear watchdog timer */
	OP_DAW,			/* decimal adjust W */
	OP_NOP,			/* no operation */
	OP_PUSH,		/* push PC onto return stack */
	OP_POP,			/* discard top of return stack */
	OP_RESET,		/* software reset */
	OP_RETFIE,		/* return from interrupt enable */
	OP_FRETFIE,		/* fast return from interrupt enable */
	OP_RETLW,		/* return with literal in W */
	OP_RET,			/* return */
	OP_FRET,		/* fast return */
	OP_SLEEP,		/* enter standby mode */
	OP_LFSR,		/* load FSR */
	OP_LBSR,		/* load BSR */
	OP_TBLRD,		/* table read */
	OP_TBLRDI,		/* table read with post-increment */
	OP_TBLRDD,		/* table read with post-decrement */
	OP_TBLIRD,		/* table read with pre-increment */
	OP_TBLWT,		/* table write */
	OP_TBLWTI,		/* table write with post-increment */
	OP_TBLWTD,		/* table write with post-decrement */
	OP_TBLIWT,		/* table write with pre-increment */
	OP_EQU,			/* define constant (no opcode) */
	OP_DW,			/* data word */
	OP_DB,			/* data byte */
	OP_ORG,			/* change origin */
	OP_NUMOPS
};


struct symbol;
struct expr;


struct inst {
	struct inst *next;
	struct symbol *sym;
#define INST_FLAG_FINAL_ARGS		0x0001
#define INST_FLAG_FINAL_LENGTH		0x0002
#define INST_FLAG_FINAL_OFFSET		0x0004
#define INST_FLAG_FINAL_INST		0x0008
#define INST_FLAG_FINAL_ALL		0x000F
#define INST_FLAG_GUESSED_OFFSET	0x0010
#define INST_FLAG_GUESSED_ARGS		0x0020
	unsigned flags;
	enum op_name op;
	unsigned offset;
	unsigned length;
	enum inst_kind kind;
	struct expr *args;
	union {
		struct {
			unsigned opcode;
			char d;
			char a;
			int freg;
		} byte;			/* IK_BYTE */
		struct {
			unsigned opcode;
			char a;
			int freg;
		} byte_file;		/* IK_BYTE_FILE */
		struct {
			unsigned opcode;
			unsigned src;
			unsigned dst;
		} byte_to_byte;		/* IK_BYTE_TO_BYTE */
		struct {
			unsigned opcode;
			unsigned bit;
			char a;
			unsigned freg;
		} bit;			/* IK_BIT */
		struct {
			unsigned opcode;
			unsigned literal;
		} op_lit;		/* IK_OP_LIT */
		struct {
			unsigned opcode;
			unsigned offset;
		} branch;		/* IK_BRANCH_* */
		struct {
			unsigned opcode;
		} oponly;		/* IK_OPONLY */
		struct {
			unsigned opcode;
			unsigned lit;
			int fsrno;
		} fsr_lit;		/* IK_FSR_LIT */
	} u;
};

struct symbol {
	struct symbol *next;
	char *label;
	struct inst *i;
};

struct token_cache {
	enum token tok;
	union {
		int num;
		struct {
			char *buf;
			int len;
		} str;
		struct symbol *sym;
	} u;
};

struct expr {
	struct expr *next;
	char literal;		/* preceded by pound */
	enum {
		vk_unknown = 0,
		vk_value,
		vk_w,
		vk_str
	} kind;
	unsigned value;
	int num_tokens;
	struct token_cache toks[MAX_TOKENS];
};

struct opinfo_rec {
	enum op_name op;
	char *name;
	int len;		/* -1 for undef */
	int num_args;		/* -1 for variable, -2 for 1 or 2 */
};


extern struct opinfo_rec opinfo[OP_NUMOPS];
extern struct inst *inst_list;

struct inst *inst_giveme(void);
void emit_inst(struct inst *i);
void dump_inst(FILE *f, struct inst *i);
struct symbol *sym_lookup(char *name);
enum op_name op_lookup(char *s);

#endif /* __INST_H__ */
