#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "main.h"

static struct symbol *curr_label = NULL;

static int include_push(const char *fn);
static int include_pop(void);
static void scan_op(void);
static struct expr *scan_args(void);
static struct expr *scan_expr(void);

static FILE *lexer_f = NULL;

void
phase1(FILE *in)
{
	lexer_f = in;

	lexer_startup();
	next_token();

	while (1) {
		if (curr_token == tok_eof) {
			fclose(lexer_f);
			lexer_f = NULL;
			if (include_pop()) {
				next_token();
				continue;
			} else {
				break;
			}
		}
		if (curr_token == tok_newline) {
			next_token();
			continue;
		}
		if (curr_token == tok_sym) {
			/* line starting with a label */
			curr_label = curr_token_sym;
			if (curr_token_sym->i) {
				do_warning("redefining '%s'\n",curr_token_sym->label);
			}
			next_token();
			scan_op();
		} else if (curr_token == tok_include) {
			/* line starting with "include" */
			next_token();
			if (curr_token != tok_str || !include_push(curr_token_str)) {
				do_warning("include directive failed, ignored\n");
				continue;
			}
			next_token();
		} else {
			/* line starting with an op */
			curr_label = NULL;
			scan_op();
		}
	}
}

struct include_stack_rec {
	struct include_stack_rec *next;
	FILE *f;
};

static struct include_stack_rec *include_stack = NULL;

/* push a file onto the include stack, true for success */
static int
include_push(const char *fn)
{
	struct include_stack_rec *p;
	FILE *f;

	f = fopen(fn,"r");
	if (!f) {
		perror(fn);
		return 0;
	}

	p = safe_malloc(sizeof *p);
	p->next = include_stack;
	p->f = lexer_f;
	include_stack = p;
	lexer_f = f;
	return 1;
}

static int
include_pop(void)
{
	struct include_stack_rec *p;

	if (!include_stack) return 0;

	p = include_stack;
	lexer_f = p->f;
	include_stack = p->next;
	free(p);
	return 1;
}

int
lex_getchar(void)
{
	return fgetc(lexer_f);
}

static void
scan_op(void)
{
	struct inst *n;
	int num_args = 0;
	struct expr *e;

	if (curr_token == tok_newline) {
		/* no instruction at all, assume it is just a label */
		if (!curr_label) return;
		n = inst_giveme();
		n->sym = curr_label;
		n->op = OP_NOP;
		n->kind = IK_LABEL;
		n->length = 0;
		n->flags |= INST_FLAG_FINAL_LENGTH | INST_FLAG_FINAL_ARGS |
				INST_FLAG_FINAL_INST;
		curr_label->i = n;
		curr_label = NULL;
		emit_inst(n);
		return;
	}

	if (curr_token != tok_op) {
		if (curr_label) {
			char *lab = curr_label->label;
			do_warning("invalid operator (%s or %s)",dump_token(curr_token), lab);
		} else {
			do_warning("invalid operator (%s)",dump_token(curr_token));
		}
		next_token();
		return;
	}

	n = inst_giveme();
	n->sym = curr_label;
	n->op = curr_token_op;
	n->kind = IK_LABEL;		/* default, no code */

	assert_str(opinfo[n->op].op == n->op, "opinfo table out of sync");

	if (curr_label) {
		curr_label->i = n;
		curr_label = NULL;
	}

	next_token();

	n->args = scan_args();

	{
		int warned = 0;
		while (curr_token != tok_newline) {
			if (!warned) {
				do_warning("ignoring trailing garbage");
				warned = 1;
			}
			next_token();
		}
	}

	for (e = n->args; e; e = e->next) {
		num_args++;
	}

	if ((num_args != opinfo[n->op].num_args) &&
	    (opinfo[n->op].num_args != -1) &&
	    ((opinfo[n->op].num_args != -2) ||
	     (num_args != 1 && num_args != 2))) {
		do_warning("got %d args to %s, expected %d", num_args, opinfo[n->op].name, opinfo[n->op].num_args);
		n->op = OP_NOP;
		n->flags |= INST_FLAG_FINAL_LENGTH | INST_FLAG_FINAL_ARGS |
				INST_FLAG_FINAL_INST;
	}

	emit_inst(n);
}

static struct expr *
scan_args(void)
{
	struct expr *args = NULL;
	struct expr *args_last = NULL;
	struct expr *n = NULL;

	while ((n = scan_expr())) {
		n->next = NULL;
		if (args_last) {
			args_last->next = n;
		} else {
			args = n;
		}
		args_last = n;
		if (curr_token == tok_newline) {
			return args;
		} else if (curr_token == tok_comma) {
			next_token();
		} else {
			break;
		}
	}
	
	return args;
}

static struct expr *
scan_expr(void)
{
	struct expr *e = NULL;
	char literal = 0;

	if (curr_token == tok_pound) {
		literal = 1;
		next_token();
	}

	while ((curr_token != tok_newline) && (curr_token != tok_comma) &&
	       (curr_token != tok_eof)) {
		if (!e) {
			e = (struct expr *) safe_malloc(sizeof *e);
			memset(e,0,sizeof *e);
			e->literal = literal;
		}
		if (e->num_tokens >= MAX_TOKENS) {
			do_warning("argument contained too many tokens");
			return e;
		}
		e->toks[e->num_tokens].tok = curr_token;
		switch (curr_token) {
			case tok_op:
				do_warning("opcode found in expression");
				return NULL;
			case tok_num:
				e->toks[e->num_tokens].u.num = curr_token_num;
				break;
			case tok_str:
				e->toks[e->num_tokens].u.str.buf = safe_malloc(curr_token_strlen+1);
				memcpy(e->toks[e->num_tokens].u.str.buf, curr_token_str, curr_token_strlen+1);
				e->toks[e->num_tokens].u.str.len = curr_token_strlen;
				break;
			case tok_sym:
				e->toks[e->num_tokens].u.sym = curr_token_sym;
				break;
			default: break;
		}
		e->num_tokens++;
		next_token();
	}
	return e;
}
