#include <stdio.h>
#include <string.h>
#include "main.h"


static void resolve_insts(void);
static void resolve_inst(struct inst *i);
static void preresolve_inst(struct inst *i);
static int resolve_args(struct expr *arg);
static unsigned scan_term(struct expr *e, int *tokidx, int *resolved);
static unsigned scan_prod(struct expr *e, int *tokidx, int *resolved);
static unsigned scan_val(struct expr *e, int *tokidx, int *resolved);
static unsigned get_value(struct inst *i, struct expr *e, char literal);
static void do_org(struct inst *i);
static void do_equ(struct inst *i);
static void do_literal(struct inst *i, int align);
static void do_oponly(struct inst *i, unsigned opcode);
static void do_branch(struct inst *i, unsigned dir_opcode, unsigned rel_opcode);
static void do_cond_branch(struct inst *i, unsigned opcode);
static void do_bit(struct inst *i, unsigned opcode);
static void do_byte(struct inst *i, unsigned opcode);
static void do_byte_w_opt(struct inst *i, unsigned opcode);
static void do_mul(struct inst *i);
static void do_mov(struct inst *i);
static void do_byte_or_lit(struct inst *i, unsigned byte_opcode, unsigned lit_opcode);
static void do_byte_file(struct inst *i, unsigned opcode);


static int any_unresolved = 0;
static unsigned curr_offset = 0;
static int curr_offset_resolved = 0;	/* 0 = invalid, 1 = guessed, 2 = final*/


void
phase2(void)
{
	int attempts = 10;

	do {
		any_unresolved = 0;
		curr_offset = 0;
		curr_offset_resolved = 2;

		resolve_insts();
	} while (any_unresolved && (attempts--));

	if (any_unresolved) {
		do_warning("some addresses are still unresolved after 10 passes");
	}
}

static void
resolve_insts(void)
{
	struct inst *i;
	for (i = inst_list; i; i = i->next) {
		if (!(i->flags & INST_FLAG_FINAL_ARGS)) {
			int res = resolve_args(i->args);
			if (res == 2) {
				i->flags |= INST_FLAG_FINAL_ARGS;
			} else if (res == 1) {
				i->flags |= INST_FLAG_GUESSED_ARGS;
			}
		}

		if (i->op == OP_ORG) {
			do_org(i);
		} else if (i->op == OP_DB) {
			/* 1-byte alignment */
		} else {
			/* 2-byte alignment is the default */
			curr_offset = (curr_offset + 1) & (~1);
		}

		if (i->op != OP_EQU) {
			if ((i->flags & INST_FLAG_FINAL_OFFSET) &&
			    (curr_offset != i->offset)) {
				assert_str(0, "final offset is waffling");
			}
			i->offset = curr_offset;
			if (curr_offset_resolved == 2) {
				i->flags |= INST_FLAG_FINAL_OFFSET;
			} else if (curr_offset_resolved == 1) {
				i->flags |= INST_FLAG_GUESSED_OFFSET;
			}
		}

		if (!(i->flags & INST_FLAG_FINAL_LENGTH) &&
		    (opinfo[i->op].len != -1)) {
			i->length = opinfo[i->op].len;
			i->flags |= INST_FLAG_FINAL_LENGTH;
		}

		if (!(i->flags & INST_FLAG_FINAL_INST) &&
		    (i->flags & (INST_FLAG_FINAL_ARGS|INST_FLAG_GUESSED_ARGS))) {
			resolve_inst(i);
		} else {
			preresolve_inst(i);
		}

		curr_offset += i->length;
		if (!(i->flags & INST_FLAG_FINAL_LENGTH)) {
			if (curr_offset_resolved > 1)
				curr_offset_resolved = 1;
		}

		if ((i->flags & INST_FLAG_FINAL_ALL) != INST_FLAG_FINAL_ALL) {
			any_unresolved++;
		}
	}
}

static void
resolve_inst(struct inst *i)
{
	switch (i->op) {
		case OP_ADD:	do_byte_or_lit(i, 0x24, 0x0F); break;
		case OP_ADDC:	do_byte(i, 0x20); break;
		case OP_AND:	do_byte_or_lit(i, 0x14, 0x0B); break;
		case OP_CLR:	do_byte_file(i, 0x6A); break;
		case OP_COMF:	do_byte_w_opt(i, 0x1C); break;
		case OP_CMPSEQ:	do_byte_file(i, 0x62); break;
		case OP_CMPSGT:	do_byte_file(i, 0x64); break;
		case OP_CMPSLT:	do_byte_file(i, 0x60); break;
		case OP_DECF:	do_byte_w_opt(i, 0x04); break;
		case OP_DECFSZ:	do_byte_w_opt(i, 0x2C);	break;
		case OP_DECFSNZ:do_byte_w_opt(i, 0x4C); break;
		case OP_INCF:	do_byte_w_opt(i, 0x28); break;
		case OP_INCFSZ:	do_byte_w_opt(i, 0x3C); break;
		case OP_INCFSNZ:do_byte_w_opt(i, 0x48); break;
		case OP_IOR:	do_byte_or_lit(i, 0x10, 0x09); break;
		case OP_MOV:	do_mov(i); break;
		case OP_MUL:	do_mul(i); break;
		case OP_NEG:	do_byte_file(i, 0x6C); break;
		case OP_RLF:	do_byte_w_opt(i, 0x44); break;
		case OP_RLFC:	do_byte_w_opt(i, 0x34); break;
		case OP_RRF:	do_byte_w_opt(i, 0x40); break;
		case OP_RRFC:	do_byte_w_opt(i, 0x30); break;
		case OP_SET:	do_byte_file(i, 0x68); break;
		case OP_SUBWB:	do_byte(i, 0x54); break;
		case OP_SUBF:	do_byte(i, 0x5C); break;
		case OP_SUBFB:	do_byte(i, 0x58); break;
		case OP_SUBLW:
			i->kind = IK_OP_LIT;
			i->u.op_lit.opcode = 0x08;
			i->u.op_lit.literal = get_value(i, i->args, 1);
			if (i->flags & INST_FLAG_FINAL_ARGS)
				i->flags |= INST_FLAG_FINAL_INST;
			break;
		case OP_SWAPF:	do_byte_w_opt(i, 0x38); break;
		case OP_TSTSZ:	do_byte_file(i, 0x66); break;
		case OP_XOR:	do_byte_or_lit(i, 0x18, 0x0A); break;
		case OP_BCLR:	do_bit(i, 0x9); break;
		case OP_BSET:	do_bit(i, 0x8); break;
		case OP_BTSC:	do_bit(i, 0xB); break;
		case OP_BTSS:	do_bit(i, 0xA); break;
		case OP_BTG:	do_bit(i, 0x7); break;
		case OP_BC:	do_cond_branch(i, 0xE2); break;
		case OP_BN:	do_cond_branch(i, 0xE6); break;
		case OP_BNC:	do_cond_branch(i, 0xE3); break;
		case OP_BNN:	do_cond_branch(i, 0xE7); break;
		case OP_BNO:	do_cond_branch(i, 0xE5); break;
		case OP_BNZ:	do_cond_branch(i, 0xE1); break;
		case OP_BO:	do_cond_branch(i, 0xE4); break;
		case OP_BZ:	do_cond_branch(i, 0xE0); break;
		case OP_GOTO:	do_branch(i, 0xEF, 0xD0); break;
		case OP_CALL:	do_branch(i, 0xEC, 0xD8); break;
		case OP_FCALL:
			i->kind = IK_BRANCH_DIRECT;
			i->u.branch.opcode = 0xED;
			i->u.branch.offset = get_value(i, i->args, 0);
			if (i->flags & INST_FLAG_FINAL_ARGS)
				i->flags |= INST_FLAG_FINAL_INST;
			break;
		case OP_CLRWDT:	do_oponly(i, 0x0004); break;
		case OP_DAW:	do_oponly(i, 0x0007); break;
		case OP_NOP:	do_oponly(i, 0x0000); break;
		case OP_PUSH:	do_oponly(i, 0x0005); break;
		case OP_POP:	do_oponly(i, 0x0006); break;
		case OP_RESET:	do_oponly(i, 0x00FF); break;
		case OP_RETFIE:	do_oponly(i, 0x0010); break;
		case OP_FRETFIE:do_oponly(i, 0x0011); break;
		case OP_RETLW:
			i->kind = IK_OP_LIT;
			i->u.op_lit.opcode = 0x0C;
			i->u.op_lit.literal = get_value(i, i->args, 1);
			if (i->flags & INST_FLAG_FINAL_ARGS)
				i->flags |= INST_FLAG_FINAL_INST;
			break;
		case OP_RET:	do_oponly(i, 0x0012); break;
		case OP_FRET:	do_oponly(i, 0x0013); break;
		case OP_SLEEP:	do_oponly(i, 0x0003); break;
		case OP_LFSR:
			i->kind = IK_FSR_LIT;
			i->u.fsr_lit.opcode = 0xEE0;
			i->u.fsr_lit.fsrno = get_value(i, i->args, 0);
			i->u.fsr_lit.lit = get_value(i, i->args->next, 0);
			if (i->flags & INST_FLAG_FINAL_ARGS)
				i->flags |= INST_FLAG_FINAL_INST;
			break;
		case OP_LBSR:
			i->kind = IK_OP_LIT;
			i->u.op_lit.opcode = 0x01;
			i->u.op_lit.literal = get_value(i, i->args, 1);
			if (i->u.op_lit.literal & (~0xF)) {
				do_warning("LBSR value out of range");
			}
			if (i->flags & INST_FLAG_FINAL_ARGS)
				i->flags |= INST_FLAG_FINAL_INST;
			break;
		case OP_TBLRD :	do_oponly(i, 0x0008); break;
		case OP_TBLRDI:	do_oponly(i, 0x0009); break;
		case OP_TBLRDD:	do_oponly(i, 0x000A); break;
		case OP_TBLIRD:	do_oponly(i, 0x000B); break;
		case OP_TBLWT :	do_oponly(i, 0x000C); break;
		case OP_TBLWTI:	do_oponly(i, 0x000D); break;
		case OP_TBLWTD:	do_oponly(i, 0x000E); break;
		case OP_TBLIWT:	do_oponly(i, 0x000F); break;
		case OP_EQU:	do_equ(i); break;
		case OP_DW:	do_literal(i, 2); break;
		case OP_DB:	do_literal(i, 1); break;
		case OP_ORG:
			/* work already done */
			i->flags |= INST_FLAG_FINAL_INST;
			break;
		default:
			assert_str(0, "unknown opcode");
	}
}


/* if an instruction's arguments aren't ready, it may still be important to,
 * for example, assign it a default "maximum length" */
static void
preresolve_inst(struct inst *i)
{
	if (i->flags & INST_FLAG_FINAL_LENGTH) {
		/* length is already set */
		return;
	}

	i->length = opinfo[i->op].len;
	if (i->length == -1) {
		/* default unresolved to largest possible */
		i->length = 4;
	}
}



/* arg = w | "string" | term
 * term = prod | prod + prod | prod - prod | prod & prod | prod | prod | prod ^ prod
 * prod = val | val * val | val / val | val % val
 * val = num | - num | sym | ( term )
 */

/* returns 2 if every arg is resolved, 1 if every arg is at least guessed */
static int
resolve_args(struct expr *arg)
{
	int resolved = 2;
	int tokidx = 0;

	if (!arg) {
		return 2;
	} else if (arg->next) {
		int t = resolve_args(arg->next);
		if (t < resolved) 
			resolved = t;
	}

	if (arg->num_tokens <= tokidx) {
		do_warning("insufficient tokens for argument, defaulting to 0");
		arg->kind = vk_value;
		arg->value = 0;
	} else {
		if (arg->toks[tokidx].tok == tok_w) {
			arg->kind = vk_w;
			tokidx++;
		} else if (arg->toks[tokidx].tok == tok_str) {
			arg->kind = vk_str;
			/* string is in arg->toks[0].u.str */
			tokidx++;
		} else {
			arg->kind = vk_value;
			arg->value = scan_term(arg, &tokidx, &resolved);
		}
		if (arg->num_tokens > tokidx) {
			do_warning("extra tokens at end of argument");
		}
	}

	return resolved;
}

static struct token_cache *
get_token(struct expr *e, int tokidx)
{
	if (tokidx >= e->num_tokens) {
		return NULL;
	}
	return &e->toks[tokidx];
}

static unsigned
scan_term(struct expr *e, int *tokidx, int *resolved)
{
	unsigned ret;
	unsigned second;
	struct token_cache *tc;

	ret = scan_prod(e, tokidx, resolved);
	tc = get_token(e, *tokidx);
	if (tc && ((tc->tok == tok_plus) || (tc->tok == tok_minus) ||
	           (tc->tok == tok_and) || (tc->tok == tok_or) ||
	           (tc->tok == tok_xor))) {
		(*tokidx)++;
		second = scan_prod(e, tokidx, resolved);
		switch (tc->tok) {
			case tok_plus: ret += second; break;
			case tok_minus: ret -= second; break;
			case tok_and: ret &= second; break;
			case tok_or: ret |= second; break;
			case tok_xor: ret ^= second; break;
			default: ;
		}
	}
	return ret;
}

static unsigned
scan_prod(struct expr *e, int *tokidx, int *resolved)
{
	unsigned ret;
	unsigned second;
	struct token_cache *tc;

	ret = scan_val(e, tokidx, resolved);
	tc = get_token(e, *tokidx);
	if (tc && ((tc->tok == tok_star) || (tc->tok == tok_slash) ||
	           (tc->tok == tok_percent))) {
		(*tokidx)++;
		second = scan_prod(e, tokidx, resolved);
		if (tc->tok == tok_star) {
			ret *= second;
		} else if (tc->tok == tok_slash) {
			ret /= second;
		} else {
			ret %= second;
		}
	}
	return ret;
}

static unsigned
scan_val(struct expr *e, int *tokidx, int *resolved)
{
	struct token_cache *tc = get_token(e, *tokidx);
	unsigned ret = 0;

	if (!tc) {
		do_warning("syntax_error");
		return 0;
	}

	if (tc->tok == tok_num) {
		ret = tc->u.num;
		(*tokidx)++;
	} else if (tc->tok == tok_minus) {
		(*tokidx)++;
		tc = get_token(e, *tokidx);
		if (tc->tok != tok_num) {
			do_warning("expecting number, got %s", dump_token(tc->tok));
			(*tokidx)--;
		} else {
			ret = -tc->u.num;
			(*tokidx)++;
		}
	} else if (tc->tok == tok_sym) {
		if (!tc->u.sym->i) {
			do_warning("undefined symbol: %s", tc->u.sym->label);
		} else {
			ret = tc->u.sym->i->offset;
			if (tc->u.sym->i->flags & INST_FLAG_FINAL_OFFSET) {
				/* happiness */
			} else if (tc->u.sym->i->flags & INST_FLAG_GUESSED_OFFSET) {
				if (*resolved > 1) *resolved = 1;
			} else {
				*resolved = 0;
			}
		}
		(*tokidx)++;
	} else if (tc->tok == tok_lparen) {
		(*tokidx)++;
		ret = scan_term(e, tokidx, resolved);
		tc = get_token(e, *tokidx);
		if (tc && (tc->tok == tok_rparen)) {
			(*tokidx)++;
		} else {
			do_warning("syntax error: expected right paren");
		}
	} else {
		do_warning("syntax error");
	}
	return ret;
}


static unsigned
get_value(struct inst *i, struct expr *e, char literal)
{
	if (!e || (e->kind != vk_value)) {
		i->flags |= INST_FLAG_FINAL_INST|INST_FLAG_FINAL_LENGTH|INST_FLAG_FINAL_ARGS;
		if (i->op == OP_EQU) {
			i->flags |= INST_FLAG_FINAL_OFFSET;
		}
		do_warning("expected value");
		return 0;
	}
	if (literal && !e->literal) {
		do_warning("expected literal, got value");
	} else if (!literal && e->literal) {
		do_warning("expected value, got literal");
	}

	return e->value;
}

static int
val_to_freg(int val, char *a, struct inst *i)
{
	if (!(val & (~0xFF))) {
		/* regular access... */
		*a = 1;
	} else if ((val & 0xF80) == 0xF80) {
		/* access bank to access high regs */
		*a = 0;
	} else if (i->flags & INST_FLAG_FINAL_ARGS) {
		do_warning("file register value %d out of range", val);
		*a = 0;
	}
	return val & 0xFF;
}


static void
do_org(struct inst *i)
{
	i->flags |= INST_FLAG_FINAL_INST;

	curr_offset = get_value(i, i->args, 0);

	if (i->flags & INST_FLAG_FINAL_ARGS) {
		curr_offset_resolved = 2;
	} else if (i->flags & INST_FLAG_GUESSED_ARGS) {
		curr_offset_resolved = 1;
	} else {
		curr_offset_resolved = 0;
	}
}

static void
do_equ(struct inst *i)
{
	i->offset = get_value(i, i->args, 0);

	if (i->flags & INST_FLAG_FINAL_ARGS) {
		i->flags |= INST_FLAG_FINAL_INST|INST_FLAG_FINAL_OFFSET;
	} else if (i->flags & INST_FLAG_GUESSED_ARGS) {
		i->flags |= INST_FLAG_GUESSED_OFFSET;
	}
}

static void
do_literal(struct inst *i, int align)
{
	struct expr *e;
	int len = 0;

	i->kind = IK_LITERAL;

	for (e = i->args; e; e = e->next) {
		if (e->kind == vk_value) {
			len += align;
		} else if (e->kind == vk_w) {
			do_warning("invalid W in literal");
		} else if (e->kind == vk_str) {
			len += e->toks[0].u.str.len;
			len = (len + (align-1)) & (~(align-1));
		}
	}

	i->length = len;
	i->flags |= INST_FLAG_FINAL_LENGTH | INST_FLAG_FINAL_INST;
}

static void
do_oponly(struct inst *i, unsigned opcode)
{
	i->kind = IK_OPONLY;
	i->u.oponly.opcode = opcode;
	i->flags |= INST_FLAG_FINAL_INST;
}

static void
do_branch(struct inst *i, unsigned dir_opcode, unsigned rel_opcode)
{
	unsigned offset = get_value(i, i->args, 0);
	int disp = (((int)offset) - ((int)(curr_offset + 2))) / 2;

	i->u.branch.offset = offset;

	if ((disp >= -1024) && (disp <= 1023)) {
		/* can try the small form */
		i->kind = IK_BRANCH_REL_LONG;
		i->u.branch.opcode = rel_opcode;
		i->length = 2;
	} else {
		i->kind = IK_BRANCH_DIRECT;
		i->u.branch.opcode = dir_opcode;
		i->length = 4;
	}

	if ((i->flags & (INST_FLAG_FINAL_ARGS|INST_FLAG_FINAL_OFFSET)) ==
	    (INST_FLAG_FINAL_ARGS|INST_FLAG_FINAL_OFFSET)) {
		i->flags |= INST_FLAG_FINAL_INST | INST_FLAG_FINAL_LENGTH;
	} else if ((i->flags & (INST_FLAG_FINAL_ARGS|INST_FLAG_GUESSED_ARGS)) &&
	    (i->flags & (INST_FLAG_FINAL_OFFSET|INST_FLAG_GUESSED_OFFSET))) {
		i->flags |= INST_FLAG_FINAL_LENGTH;
	} else {
		i->length = 4;
	}
}

static void
do_cond_branch(struct inst *i, unsigned opcode)
{
	unsigned offset = get_value(i, i->args, 0);

	i->kind = IK_BRANCH_REL_SHORT;
	i->u.branch.opcode = opcode;
	i->u.branch.offset = offset;

	if (i->flags & INST_FLAG_FINAL_ARGS)
		i->flags |= INST_FLAG_FINAL_INST;
}

static void
do_bit(struct inst *i, unsigned opcode)
{
	int arg = get_value(i, i->args, 0);

	i->kind = IK_BIT;
	i->u.bit.opcode = opcode;

	if (i->args->next) {
		i->u.bit.bit = get_value(i, i->args->next, 0);
	} else {
		i->u.bit.bit = arg % 8;
		arg /= 8;
	}

	i->u.bit.freg = val_to_freg(arg, &i->u.bit.a, i);

	if (i->flags & INST_FLAG_FINAL_ARGS)
		i->flags |= INST_FLAG_FINAL_INST;
}

static void
do_byte(struct inst *i, unsigned opcode)
{
	int reg;

	i->kind = IK_BYTE;
	i->u.byte.opcode = opcode;

	if ((i->args->kind == vk_w) &&
	    (i->args->next->kind == vk_value)) {
		i->u.byte.d = 0;
		reg = i->args->next->value;
	} else if ((i->args->kind == vk_value) &&
		   (i->args->next->kind == vk_w)) {
		i->u.byte.d = 1;
		reg = i->args->value;
	} else {
		do_warning("BYTE opcodes can only use one file reg and W");
		reg = 0;
		i->flags |= INST_FLAG_FINAL_INST|INST_FLAG_FINAL_ARGS;
	}

	i->u.byte.freg = val_to_freg(reg, &i->u.byte.a, i);

	if (i->flags & INST_FLAG_FINAL_ARGS)
		i->flags |= INST_FLAG_FINAL_INST;
}

static void
do_byte_w_opt(struct inst *i, unsigned opcode)
{
	int reg;

	i->kind = IK_BYTE;
	i->u.byte.opcode = opcode;

	if ((i->args->kind == vk_w) &&
	    i->args->next &&
	    (i->args->next->kind == vk_value)) {
		i->u.byte.d = 0;
		reg = i->args->next->value;
	} else if ((i->args->kind == vk_value) &&
		   !i->args->next) {
		i->u.byte.d = 1;
		reg = i->args->value;
	} else {
		do_warning("BYTE opcodes can only use one file reg and W");
		reg = 0;
		i->flags |= INST_FLAG_FINAL_INST|INST_FLAG_FINAL_ARGS;
	}

	i->u.byte.freg = val_to_freg(reg, &i->u.byte.a, i);

	if (i->flags & INST_FLAG_FINAL_ARGS)
		i->flags |= INST_FLAG_FINAL_INST;
}

static void
do_mul(struct inst *i)
{
	struct expr *e;

	if ((i->args->kind == vk_w) &&
	    (i->args->next->kind == vk_value)) {
		e = i->args->next;
	} else if ((i->args->kind == vk_value) &&
		   (i->args->next->kind == vk_w)) {
		e = i->args;
	} else {
		do_warning("MUL can only use one file reg or literal and W");
		i->flags |= INST_FLAG_FINAL_INST|INST_FLAG_FINAL_ARGS;
		i->kind = IK_LABEL;
		return;
	}

	if (e->literal) {
		i->kind = IK_OP_LIT;
		i->u.op_lit.opcode = 0x0D;
		i->u.op_lit.literal = e->value;
	} else {
		i->kind = IK_BYTE_FILE;
		i->u.byte_file.opcode = 0x02;
		i->u.byte_file.freg = val_to_freg(e->value, &i->u.byte_file.a, i);
	}

	if (i->flags & INST_FLAG_FINAL_ARGS)
		i->flags |= INST_FLAG_FINAL_INST;
}

static void
do_mov(struct inst *i)
{
	if ((i->args->kind == vk_w) &&
	    (i->args->next->kind == vk_value)) {
		int arg = i->args->next->value;
		if (i->args->next->literal) {		/* MOVLW */
			i->kind = IK_OP_LIT;
			i->u.op_lit.opcode = 0x0E;
			i->u.op_lit.literal = arg;
			i->length = 2;
		} else {				/* MOVF */
			i->kind = IK_BYTE;
			i->u.byte.opcode = 0x50;
			i->u.byte.d = 0;
			i->u.byte.freg = val_to_freg(arg, &i->u.byte.a, i);
			i->length = 2;
		}
	} else if ((i->args->kind == vk_value) &&
		   (i->args->next->kind == vk_w)) {	/* MOVWF */
		int arg = get_value(i, i->args, 0);
		i->kind = IK_BYTE_FILE;
		i->u.byte_file.opcode = 0x6E;
		i->u.byte_file.freg = val_to_freg(arg, &i->u.byte_file.a, i);
		i->length = 2;
	} else if ((i->args->kind == vk_value) &&
		   (i->args->next->kind == vk_value)) {
		int lhs = get_value(i, i->args, 0);
		int rhs = get_value(i, i->args->next, 0);
		if ((lhs == rhs) && 
		    (((lhs & ~0xFF) == 0) ||
		     ((lhs & ~0xFF) == 0xF00))) {	/* MOVF */
			i->kind = IK_BYTE;
			i->u.byte.opcode = 0x50;
			i->u.byte.d = 1;
			i->u.byte.freg = val_to_freg(lhs, &i->u.byte.a, i);
			i->length = 2;
		} else {				/* MOVFF */
			i->kind = IK_BYTE_TO_BYTE;
			i->u.byte_to_byte.opcode = 0xC;
			i->u.byte_to_byte.src = rhs;
			i->u.byte_to_byte.dst = lhs;
			i->length = 4;
		}
	} else {
		do_warning("MOV can only use  W,f  f,W  f,f  or  W,lit");
	}
	if (i->flags & (INST_FLAG_FINAL_ARGS|INST_FLAG_GUESSED_ARGS)) {
		i->flags |= INST_FLAG_FINAL_LENGTH;
	} else {
		i->length = 4;
	}
	if (i->flags & INST_FLAG_FINAL_ARGS) {
		i->flags |= INST_FLAG_FINAL_INST;
	}
}

static void
do_byte_or_lit(struct inst *i, unsigned byte_opcode, unsigned lit_opcode)
{
	if ((i->args->kind == vk_w) &&
	    (i->args->next->kind == vk_value)) {
		int arg = i->args->next->value;
		if (i->args->next->literal) {
			i->kind = IK_OP_LIT;
			i->u.op_lit.opcode = lit_opcode;
			i->u.op_lit.literal = arg;
		} else {
			i->kind = IK_BYTE;
			i->u.byte.opcode = byte_opcode;
			i->u.byte.d = 0;
			i->u.byte.freg = val_to_freg(arg, &i->u.byte.a, i);
		}
	} else if ((i->args->kind == vk_value) &&
		   (i->args->next->kind == vk_w)) {
		int arg = get_value(i, i->args, 0);
		i->kind = IK_BYTE;
		i->u.byte.opcode = byte_opcode;
		i->u.byte.d = 1;
		i->u.byte.freg = val_to_freg(arg, &i->u.byte.a, i);
	} else {
		do_warning("BYTE/LITERAL opcodes can only use one file reg or literal and W");
		i->flags |= INST_FLAG_FINAL_INST|INST_FLAG_FINAL_ARGS;
		i->kind = IK_LABEL;
		return;
	}

	if (i->flags & INST_FLAG_FINAL_ARGS)
		i->flags |= INST_FLAG_FINAL_INST;
}

static void
do_byte_file(struct inst *i, unsigned opcode)
{
	int arg = get_value(i, i->args, 0);

	i->kind = IK_BYTE_FILE;
	i->u.byte_file.opcode = opcode;

	i->u.byte_file.freg = val_to_freg(arg, &i->u.byte_file.a, i);

	if (i->flags & INST_FLAG_FINAL_ARGS)
		i->flags |= INST_FLAG_FINAL_INST;
}
