#include <stdio.h>
#include <string.h>
#include "main.h"


#define IMAGE_NUM_BYTES 8192
#define CONFIG_OFFSET 0x300000
#define CONFIG_BYTES 0x0E
#define DATA_OFFSET 0xF00000
#define DATA_BYTES 0x200


static void output_hex_file(FILE *f);
static void output_special_area(FILE *f, unsigned char *bytes, unsigned char *set, unsigned len, unsigned offset);
static void output_hex_line(FILE *f, unsigned char type, unsigned addr, unsigned char *data, unsigned len);
static void output_inst(struct inst *i);
static void list_inst(FILE *f, struct inst *i, int start, int end);


static unsigned char image[IMAGE_NUM_BYTES];
static struct inst *image_i[IMAGE_NUM_BYTES];
static unsigned char config_bytes[CONFIG_BYTES];
static unsigned char config_set[CONFIG_BYTES];
static unsigned char data_bytes[DATA_BYTES];
static unsigned char data_set[DATA_BYTES];
static unsigned image_highest;


void
phase3(FILE *bin, FILE *hex, FILE *list)
{
	struct inst *i;

	memset(image, 0, sizeof image);
	image_highest = 0;

	for (i = inst_list; i; i = i->next) {
		output_inst(i);
	}

	if (bin) {
		fwrite(image, image_highest+1, 1, bin);
	}

	if (list) {
		int j;
		int when_i = 0;
		i = NULL;
		for (j = 0; j <= image_highest; j++) {
			if (i != image_i[j]) {
				list_inst(list, i, when_i, j);
				when_i = j;
				i = image_i[j];
			}
		}
		list_inst(list, i, when_i, j);
		for (j = 0; j < CONFIG_BYTES; j++) {
			if (config_set[j]) 
				fprintf(list,"%08X  %02X\n",CONFIG_OFFSET+j,config_bytes[j]);
		}
		for (j = 0; j < DATA_BYTES; j++) {
			if (data_set[j])
				fprintf(list,"%08X  %02X\n",DATA_OFFSET+j,data_bytes[j]);
		}
	}

	if (hex) {
		output_hex_file(hex);
	}
}


static void
output_hex_file(FILE *f)
{
	int i, start = 0, start_has_something = 0;

		/* set base address to 0x0000 0000 */
	output_hex_line(f, 0x04, 0, "\000\000", 2);

	for (i = 0; i < image_highest; i+=2) {
		if (i-start >= 16) {
			output_hex_line(f, 0x00, start, &image[start], i-start);
			start = i;
			start_has_something = 0;
		}
		if (!image_i[i] && !image_i[i+1]) {
			if (start_has_something) {
				output_hex_line(f, 0x00, start, &image[start], i-start);
				start = i;
				start_has_something = 0;
			}
		} else {
			if (!start_has_something) {
				start = i;
				start_has_something = 1;
			}
		}
	}
	if (start_has_something) {
		output_hex_line(f, 0x00, start, &image[start], i-start);
	}

	output_special_area(f, config_bytes, config_set, CONFIG_BYTES, CONFIG_OFFSET);
	output_special_area(f, data_bytes, data_set, DATA_BYTES, DATA_OFFSET);

	output_hex_line(f, 0x01, 0, "", 0);
}

static void
output_special_area(FILE *f, unsigned char *bytes, unsigned char *set, unsigned len, unsigned offset)
{
	int i, start = 0, start_has_something = 0;
	char buf[10];

	buf[0] = (offset >> 24) & 0xFF;
	buf[1] = (offset >> 16) & 0xFF;

		/* set base address */
	output_hex_line(f, 0x04, 0, buf, 2);

	for (i = 0; i < len; i+=2) {
		if (!set[i] && !set[i+1]) {
			if (start_has_something) {
				output_hex_line(f, 0x00, start, &bytes[start], i-start);
				start = i;
				start_has_something = 0;
			}
		} else {
			if (!start_has_something) {
				start = i;
				start_has_something = 1;
			}
		}
	}
	if (start_has_something) {
		output_hex_line(f, 0x00, start, &bytes[start], i-start);
	}
}

/* outputs a hex line corresponding to len bytes of data at addr of type,
 * optionally byte swapped because that seems to be needed for program
 * memory.  len should be a multiple of two less than 16. */
static void
output_hex_line(FILE *f, unsigned char type, unsigned addr, unsigned char *data, unsigned len)
{
	int cksum = 0;
	char buf[40];
	int i;

	assert_str(len<=16, "too many bytes at once in output_hex_line()");

	cksum = (cksum + len) & 0xFF;
	cksum = (cksum + ((addr&0xFF00) >> 8)) & 0xFF;
	cksum = (cksum + (addr&0x00FF)) & 0xFF;
	cksum = (cksum + type) & 0xFF;
	buf[0] = 0;
	for (i = 0; i < len; i++) {
		unsigned char c = data[i];
		sprintf(&buf[i*2], "%02X", c);
		cksum = (cksum + c) & 0xFF;
	}
	fprintf(f, ":%02X%04X%02X%s%02X\n", len, addr, type, buf, ((cksum^0xFF)+1)&0xFF);
}

static unsigned bits;
static unsigned bits_mask;
static unsigned bits_swap;		/* if the endian needs to be swapped */

static void
copy_bits(unsigned src, char *name, int ofs, int len)
{
	int mask = (1 << len)-1;
	int sh = (16 - len) - ofs;
	if (((src & ~mask) != 0) && ((src & ~mask) != ~mask)) {
		do_warning("%s field of instruction exceeds %d bits (%d)", name, len, src);
	}
	if (bits_mask & (mask << sh)) {
		do_warning("%s field overlaps existing data in instruction", name);
	}
	bits_mask |= (mask << sh);
	bits |= ((src & mask) << sh);
}

static void
output_bits(struct inst *i, unsigned offset)
{
	unsigned len = 2;

	if (bits_mask == 0xFF00) {
		/* output just the first character */
		len = 1;
	} else if (bits_mask == 0xFFFF) {
		len = 2;
		if (bits_swap) {
			bits = ((bits & 0xFF) << 8) | ((bits & 0xFF00) >> 8);
		}
	} else {
		do_warning("some undefined bits in output code");
	}

	if (offset >= CONFIG_OFFSET && offset + len <= (CONFIG_OFFSET+CONFIG_BYTES)) {
		/* configuration range */
		int o = offset-CONFIG_OFFSET;
		config_bytes[o] = bits>>8;
		if (config_set[o]) {
			do_warning("output byte %02X replacing %02X at %08X", bits>>8, config_bytes[o], offset);
		}
		config_set[o]++;
		if (len == 2) {
			config_bytes[o+1] =  bits&0xFF;
			if (config_set[o+1]) {
				do_warning("output byte %02X replacing %02X at %08X", bits&0xFF, config_bytes[o+1], offset+1);
			}
			config_set[o+1]++;
		}
		bits = 0;
		bits_mask = 0;
		return;
	} else if (offset >= DATA_OFFSET && offset + len <= (DATA_OFFSET+DATA_BYTES)) {
		/* configuration range */
		int o = offset-DATA_OFFSET;
		data_bytes[o] = bits>>8;
		if (data_set[o]) {
			do_warning("output byte %02X replacing %02X at %08X", bits>>8, data_bytes[o], offset);
		}
		data_set[o]++;
		if (len == 2) {
			data_bytes[o+1] =  bits&0xFF;
			if (data_set[o+1]) {
				do_warning("output byte %02X replacing %02X at %08X", bits&0xFF, data_bytes[o+1], offset+1);
			}
			data_set[o+1]++;
		}
		bits = 0;
		bits_mask = 0;
		return;
	} else if (offset + len > IMAGE_NUM_BYTES) {
		do_warning("output data out of range (%08X)",offset);
		return;
	}

	if (image[offset]) {
		do_warning("output byte %02X replacing %02X at %08X", bits>>8, image[offset], offset);
	}
	image[offset] = bits>>8;
	image_i[offset] = i;

	if (len == 2) {	/* do the second byte */
		offset ++;
		if (image[offset]) {
			do_warning("output byte %02X replacing %02X at %08X", bits&0xFF, image[offset], offset);
		}
		image[offset] = bits&0xFF;
		image_i[offset] = i;
	}

	if (offset > image_highest) image_highest = offset;
	
	bits = 0;
	bits_mask = 0;
}

static void
output_literal(struct inst *i)
{
	int align = ((i->op == OP_DB) ? 1 : 2);
	unsigned offset = i->offset;
	struct expr *e;

	bits_swap = 0;

	for (e = i->args; e; e = e->next) {
		if (e->kind == vk_value) {
			if (align == 1) {
				copy_bits(e->value, "DB", 0, 8);
			}  else {	/* align == 2 */
				copy_bits(e->value>>8,   "DWh", 0, 8);
				copy_bits(e->value&0xFF, "DWl", 8, 8);
			}
			output_bits(i, offset);
			offset += align;
		} else if (e->kind == vk_str) {
			unsigned char *s;
			int n,len;
			s = e->toks[0].u.str.buf;
			len = e->toks[0].u.str.len;
			for (n = 0; n < len; n++) {
				copy_bits(s[n], "str", 0, 8);
				output_bits(i, offset);
				offset++;
			}
			offset = (offset + (align-1)) & ~(align-1);
		}
	}
}

static void
output_inst(struct inst *i)
{
	bits = 0;
	bits_mask = 0;
	bits_swap = 1;

	switch (i->kind) {
		case IK_LABEL:
			/* nothing to emit */
			break;
		case IK_LITERAL:
			output_literal(i);
			break;
		case IK_BYTE:
			copy_bits(i->u.byte.opcode>>2,	"op",	0, 6);
			copy_bits(i->u.byte.d,		"d",	6, 1);
			copy_bits(i->u.byte.a,		"a",	7, 1);
			copy_bits(i->u.byte.freg,	"freg",	8, 8);
			output_bits(i, i->offset);
			break;
		case IK_BYTE_FILE:
			copy_bits(i->u.byte_file.opcode>>1,"op",0, 7);
			copy_bits(i->u.byte_file.a,	"a",	7, 1);
			copy_bits(i->u.byte_file.freg,	"freg",	8, 8);
			output_bits(i, i->offset);
			break;
		case IK_BYTE_TO_BYTE:
			copy_bits(i->u.byte_to_byte.opcode,	"op",	0, 4);
			copy_bits(i->u.byte_to_byte.src,	"src",	4, 12);
			output_bits(i, i->offset);
			copy_bits(0xF,				"-1",	0, 4);
			copy_bits(i->u.byte_to_byte.dst,	"dst",	4, 12);
			output_bits(i, i->offset+2);
			break;
		case IK_BIT:
			copy_bits(i->u.bit.opcode,	"op",	0, 4);
			copy_bits(i->u.bit.bit,		"b",	4, 3);
			copy_bits(i->u.bit.a,		"a",	7, 1);
			copy_bits(i->u.bit.freg,	"freg",	8, 8);
			output_bits(i, i->offset);
			break;
		case IK_OP_LIT:
			copy_bits(i->u.op_lit.opcode,	"op",	0, 8);
			if ((i->u.op_lit.literal&0xFF80U) == 0xFF80U) {
					/* probably a reasonable negative
					 * value, convert it to 8-bit signed */
				i->u.op_lit.literal &= 0xFF;
			}
			copy_bits(i->u.op_lit.literal,	"lit",	8, 8);
			output_bits(i, i->offset);
			break;
		case IK_BRANCH_DIRECT: {
			unsigned addr = i->u.branch.offset >> 1;
			copy_bits(i->u.branch.opcode,	"op",	0, 8);
			copy_bits(addr & 0xFF,		"addrl",8, 8);
			output_bits(i, i->offset);
			copy_bits(0xF,			"-1",	0, 4);
			copy_bits(addr >> 8,		"addrh",4, 12);
			output_bits(i, i->offset+2);
		}	break;
		case IK_BRANCH_REL_SHORT: {
			int ofs = (((int)i->u.branch.offset) - ((int)(i->offset+2))) / 2;
			copy_bits(i->u.branch.opcode,	"op",	0, 8);
			copy_bits(ofs,			"ofs",	8, 8);
			output_bits(i, i->offset);
		}	break;
		case IK_BRANCH_REL_LONG: {
			int ofs = (((int)i->u.branch.offset) - ((int)(i->offset+2))) / 2;
			copy_bits(i->u.branch.opcode>>3,"op",	0, 5);
			copy_bits(ofs,			"ofs",	5, 11);
			output_bits(i, i->offset);
		}	break;
		case IK_OPONLY:
			copy_bits(i->u.oponly.opcode,	"op",	0, 16);
			output_bits(i, i->offset);
			break;
		case IK_FSR_LIT:
			copy_bits(i->u.fsr_lit.opcode>>2,"op",	0, 10);
			copy_bits(i->u.fsr_lit.fsrno,	"fsr",	10, 2);
			copy_bits(i->u.fsr_lit.lit>>8,	"lith",12,4);
			output_bits(i, i->offset);
			copy_bits(0xF,			"-1",	0, 4);
			copy_bits(0x0,			"0",	4, 4);
			copy_bits(i->u.fsr_lit.lit&0xFF,"litl",8,8);
			output_bits(i, i->offset+2);
			break;
		default:
			assert_str(0, "unknown inst kind in phase3");
	}
}

/* produce a listing line for the instructing starting at start and
 * ending at end-1 */
static void
list_inst(FILE *f, struct inst *i, int start, int end)
{
	char buf[100];
	int j;

	if (!i) return;

	sprintf(buf, "%08X  ",start);
	for (j = start; j < end && j < start+8; j++) {
		sprintf(buf+strlen(buf), "%02X ", image[j]);
	}
	fprintf(f, "%-40.40s ",buf);
	dump_inst(f, i);
	fprintf(f, "\n");
}
